<?php

namespace Bricks;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

class Brf_Add_To_Calendar_Button extends Element
{
    public $block    = 'core/button';
    public $category = 'basic';
    public $name     = 'brf-add-to-calender-button';
    public $icon     = 'ti-calendar';
    public $tag      = 'span';
    public $nestable = true;

    public function get_label()
    {
        return esc_html__('Add To Calendar Button', 'bricksforge');
    }

    public function enqueue_scripts()
    {
        wp_enqueue_script('bricksforge-elements');
    }

    public function set_control_groups()
    {
        $this->control_groups['general'] = [
            'title' => esc_html__('General', 'bricksforge'),
            'tab'   => 'content',
        ];

        $this->control_groups['icon'] = [
            'title' => esc_html__('Button Icon', 'bricksforge'),
            'tab'   => 'content',
        ];

        $this->control_groups['events'] = [
            'title' => esc_html__('Events', 'bricksforge'),
            'tab'   => 'content',
        ];

        $this->control_groups['calendar'] = [
            'title' => esc_html__('Calendar Options', 'bricksforge'),
            'tab'   => 'content',
        ];

        $this->control_groups['dropdownIcons'] = [
            'title' => esc_html__('Dropdown Icons', 'bricksforge'),
            'tab'   => 'content',
            'required' => ['calendarType', '=', 'multiple'],
        ];

        $this->control_groups['dropdown'] = [
            'title' => esc_html__('Multiple Options Styling', 'bricksforge'),
            'tab'   => 'content',
            'required' => ['calendarType', '=', 'multiple'],
        ];
    }

    public function set_controls()
    {
        $this->controls['tag'] = [
            'label'          => esc_html__('HTML tag', 'bricksforge'),
            'type'           => 'text',
            'hasDynamicData' => false,
            'inline'         => true,
            'placeholder'    => 'span',
            'required'       => ['link', '=', ''],
            'default'        => 'a',
            'group'          => 'general',
        ];

        $this->controls['size'] = [
            'label'       => esc_html__('Size', 'bricksforge'),
            'type'        => 'select',
            'options'     => $this->control_options['buttonSizes'],
            'inline'      => true,
            'reset'       => true,
            'placeholder' => esc_html__('Default', 'bricksforge'),
            'group'       => 'general',
        ];

        $this->controls['style'] = [
            'label'       => esc_html__('Style', 'bricksforge'),
            'type'        => 'select',
            'options'     => $this->control_options['styles'],
            'inline'      => true,
            'reset'       => true,
            'default'     => 'primary',
            'placeholder' => esc_html__('None', 'bricksforge'),
            'group'       => 'general',
        ];

        $this->controls['circle'] = [
            'label' => esc_html__('Circle', 'bricksforge'),
            'type'  => 'checkbox',
            'reset' => true,
            'group' => 'general',
        ];

        $this->controls['outline'] = [
            'label' => esc_html__('Outline', 'bricksforge'),
            'type'  => 'checkbox',
            'reset' => true,
            'group' => 'general',
        ];

        $this->controls['cssClass'] = [
            'label' => esc_html__('CSS Class', 'bricksforge'),
            'type'  => 'text',
            'group' => 'general',
        ];

        // Events Repeater
        $this->controls['events'] = [
            'label'          => esc_html__('Events', 'bricksforge'),
            'type'           => 'repeater',
            'titleProperty'  => 'eventTitle',
            'placeholder'    => esc_html__('Event Title', 'bricksforge'),
            'group'          => 'events',
            'default'        => [
                [
                    'eventTitle' => esc_html__('My Event', 'bricksforge'),
                    'startDate' => '',
                    'startTime' => '10:00',
                    'endTime' => '17:00',
                    'allDay' => true,
                    'enableReminder' => false,
                ],
            ],
            'fields' => [
                'eventTitle' => [
                    'label'          => esc_html__('Event Title', 'bricksforge'),
                    'type'           => 'text',
                    'hasDynamicData' => true,
                    'placeholder'    => esc_html__('Enter event title', 'bricksforge'),
                ],
                'eventDescription' => [
                    'label'          => esc_html__('Event Description', 'bricksforge'),
                    'type'           => 'textarea',
                    'hasDynamicData' => true,
                    'placeholder'    => esc_html__('Enter event description', 'bricksforge'),
                ],
                'eventLocation' => [
                    'label'          => esc_html__('Event Location', 'bricksforge'),
                    'type'           => 'text',
                    'hasDynamicData' => true,
                    'placeholder'    => esc_html__('Enter event location', 'bricksforge'),
                ],
                'eventUrl' => [
                    'label'          => esc_html__('Event URL', 'bricksforge'),
                    'type'           => 'text',
                    'hasDynamicData' => true,
                    'placeholder'    => esc_html__('https://example.com', 'bricksforge'),
                ],
                // Static, Dynamic Data
                'eventDateMethod' => [
                    'label' => esc_html__('Event Date Method', 'bricksforge'),
                    'type'  => 'select',
                    'options' => [
                        'static' => esc_html__('Static', 'bricksforge'),
                        'dynamic' => esc_html__('Dynamic Data', 'bricksforge'),
                    ],
                    'default' => 'static',
                ],
                'allDay' => [
                    'label' => esc_html__('All Day Event', 'bricksforge'),
                    'type'  => 'checkbox',
                ],
                'startDate' => [
                    'label'          => esc_html__('Start Date', 'bricksforge'),
                    'type'           => 'datepicker',
                    'hasDynamicData' => true,
                    'placeholder'    => esc_html__('Select start date', 'bricksforge'),
                    'inline'         => true,
                    'required'       => ['eventDateMethod', '=', 'static'],
                ],
                'startTime' => [
                    'label'          => esc_html__('Start Time', 'bricksforge'),
                    'type'           => 'text',
                    'hasDynamicData' => true,
                    'placeholder'    => '09:00',
                    'description'    => esc_html__('Format: HH:MM (24-hour format)', 'bricksforge'),
                    'inline'         => true,
                    'required'       => [["allDay", "!=", true], ["eventDateMethod", "=", "static"]],

                ],
                'endDate' => [
                    'label'          => esc_html__('End Date', 'bricksforge'),
                    'type'           => 'datepicker',
                    'hasDynamicData' => true,
                    'placeholder'    => esc_html__('Select end date (optional)', 'bricksforge'),
                    'inline'         => true,
                    'required'       => ['eventDateMethod', '=', 'static'],
                ],
                'endTime' => [
                    'label'          => esc_html__('End Time', 'bricksforge'),
                    'type'           => 'text',
                    'hasDynamicData' => true,
                    'placeholder'    => '17:00',
                    'description'    => esc_html__('Format: HH:MM (24-hour format)', 'bricksforge'),
                    'inline'         => true,
                    'required'       => ['allDay', '=', false],
                    'required'       => ['eventDateMethod', '=', 'static'],
                ],
                // Dynamic Data: Start Date, End Date, Start Time, End Time
                'startDateDynamic' => [
                    'label' => esc_html__('Start Datetime', 'bricksforge'),
                    'type'  => 'text',
                    'hasDynamicData' => true,
                    'placeholder'    => esc_html__('Enter start datetime', 'bricksforge'),
                    'required'       => ['eventDateMethod', '=', 'dynamic'],
                    'description'    => esc_html__('Enter the dynamic data for the start datetime. Format: YYYY-MM-DD HH:MM:SS', 'bricksforge'),
                ],
                'endDateDynamic' => [
                    'label' => esc_html__('End Datetime', 'bricksforge'),
                    'type'  => 'text',
                    'hasDynamicData' => true,
                    'placeholder'    => esc_html__('Enter end datetime', 'bricksforge'),
                    'required'       => ['eventDateMethod', '=', 'dynamic'],
                    'description'    => esc_html__('Enter the dynamic data for the end datetime. Format: YYYY-MM-DD HH:MM:SS', 'bricksforge'),
                ],
                'enableReminder' => [
                    'label' => esc_html__('Enable Reminder', 'bricksforge'),
                    'type'  => 'checkbox',
                ],
                'reminderMinutes' => [
                    'label'   => esc_html__('Reminder Time', 'bricksforge'),
                    'type'    => 'select',
                    'options' => [
                        '0'     => esc_html__('At event time', 'bricksforge'),
                        '5'     => esc_html__('5 minutes before', 'bricksforge'),
                        '10'    => esc_html__('10 minutes before', 'bricksforge'),
                        '15'    => esc_html__('15 minutes before', 'bricksforge'),
                        '30'    => esc_html__('30 minutes before', 'bricksforge'),
                        '60'    => esc_html__('1 hour before', 'bricksforge'),
                        '120'   => esc_html__('2 hours before', 'bricksforge'),
                        '1440'  => esc_html__('1 day before', 'bricksforge'),
                        '2880'  => esc_html__('2 days before', 'bricksforge'),
                        '10080' => esc_html__('1 week before', 'bricksforge'),
                    ],
                    'default' => '15',
                    'required' => ['enableReminder', '=', true],
                ],
            ],
        ];

        $this->controls['timezone'] = [
            'label'       => esc_html__('Timezone', 'bricksforge'),
            'type'        => 'select',
            'options'     => $this->get_timezone_options(),
            'default'     => $this->get_user_timezone(),
            'placeholder' => esc_html__('Select timezone', 'bricksforge'),
            'group'       => 'events',
        ];

        // Calendar Options Controls
        $this->controls['calendarType'] = [
            'label'   => esc_html__('Calendar Type', 'bricksforge'),
            'type'    => 'select',
            'options' => [
                'google'   => esc_html__('Google Calendar', 'bricksforge'),
                'outlook'  => esc_html__('Outlook', 'bricksforge'),
                'apple'    => esc_html__('Apple Calendar', 'bricksforge'),
                'ics'      => esc_html__('Download ICS File', 'bricksforge'),
                'multiple' => esc_html__('Show Multiple Options', 'bricksforge'),
            ],
            'default' => 'ics',
            'inline'  => true,
            'group'   => 'calendar',
        ];

        $this->controls['filename'] = [
            'label'          => esc_html__('ICS Filename', 'bricksforge'),
            'type'           => 'text',
            'hasDynamicData' => true,
            'placeholder'    => 'event',
            'description'    => esc_html__('Filename for the downloaded ICS file (without extension)', 'bricksforge'),
            'required'       => ['calendarType', '=', ['ics', 'multiple']],
            'group'          => 'calendar',
        ];



        // Multiple Options Styling Controls
        $this->controls['dropdownBackground'] = [
            'label' => esc_html__('Dropdown Background', 'bricksforge'),
            'type'  => 'background',
            'css'   => [
                [
                    'property' => 'background',
                    'selector' => '.brf-calendar-options .brf-calendar-dropdown',
                ],
            ],
            'required' => ['calendarType', '=', 'multiple'],
            'group'    => 'dropdown',
        ];

        $this->controls['dropdownBorder'] = [
            'label' => esc_html__('Dropdown Border', 'bricksforge'),
            'type'  => 'border',
            'css'   => [
                [
                    'property' => 'border',
                    'selector' => '.brf-calendar-options .brf-calendar-dropdown',
                ],
            ],
            'required' => ['calendarType', '=', 'multiple'],
            'group'    => 'dropdown',
        ];

        $this->controls['dropdownBoxShadow'] = [
            'label' => esc_html__('Dropdown Box Shadow', 'bricksforge'),
            'type'  => 'box-shadow',
            'css'   => [
                [
                    'property' => 'box-shadow',
                    'selector' => '.brf-calendar-options .brf-calendar-dropdown',
                ],
            ],
            'required' => ['calendarType', '=', 'multiple'],
            'group'    => 'dropdown',
        ];

        $this->controls['dropdownPadding'] = [
            'label' => esc_html__('Dropdown Padding', 'bricksforge'),
            'type'  => 'spacing',
            'css'   => [
                [
                    'property' => 'padding',
                    'selector' => '.brf-calendar-options .brf-calendar-dropdown',
                ],
            ],
            'required' => ['calendarType', '=', 'multiple'],
            'group'    => 'dropdown',
        ];

        $this->controls['dropdownWidth'] = [
            'label' => esc_html__('Dropdown Width', 'bricksforge'),
            'type'  => 'number',
            'units' => true,
            'css'   => [
                [
                    'property' => 'width',
                    'selector' => '.brf-calendar-options .brf-calendar-dropdown',
                ],
            ],
            'required' => ['calendarType', '=', 'multiple'],
            'group'    => 'dropdown',
        ];

        $this->controls['optionBackground'] = [
            'label' => esc_html__('Option Background', 'bricksforge'),
            'type'  => 'background',
            'css'   => [
                [
                    'property' => 'background',
                    'selector' => '.brf-calendar-options .brf-calendar-option',
                ],
            ],
            'required' => ['calendarType', '=', 'multiple'],
            'group'    => 'dropdown',
        ];

        $this->controls['optionHoverBackground'] = [
            'label' => esc_html__('Option Hover Background', 'bricksforge'),
            'type'  => 'background',
            'css'   => [
                [
                    'property' => 'background',
                    'selector' => '.brf-calendar-options .brf-calendar-option:hover',
                ],
            ],
            'required' => ['calendarType', '=', 'multiple'],
            'group'    => 'dropdown',
        ];

        $this->controls['optionTypography'] = [
            'label' => esc_html__('Option Typography', 'bricksforge'),
            'type'  => 'typography',
            'css'   => [
                [
                    'property' => 'font',
                    'selector' => '.brf-calendar-options .brf-calendar-option',
                ],
            ],
            'required' => ['calendarType', '=', 'multiple'],
            'group'    => 'dropdown',
        ];

        $this->controls['optionPadding'] = [
            'label' => esc_html__('Option Padding', 'bricksforge'),
            'type'  => 'spacing',
            'css'   => [
                [
                    'property' => 'padding',
                    'selector' => '.brf-calendar-options .brf-calendar-option',
                ],
            ],
            'required' => ['calendarType', '=', 'multiple'],
            'group'    => 'dropdown',
        ];

        $this->controls['optionBorder'] = [
            'label' => esc_html__('Option Border', 'bricksforge'),
            'type'  => 'border',
            'css'   => [
                [
                    'property' => 'border',
                    'selector' => '.brf-calendar-options .brf-calendar-option',
                ],
            ],
            'required' => ['calendarType', '=', 'multiple'],
            'group'    => 'dropdown',
        ];

        $this->controls['optionGap'] = [
            'label' => esc_html__('Option Gap', 'bricksforge'),
            'type'  => 'number',
            'units' => true,
            'css'   => [
                [
                    'property' => 'gap',
                    'selector' => '.brf-calendar-options .brf-calendar-option',
                ],
            ],
            'required' => ['calendarType', '=', 'multiple'],
            'group'    => 'dropdown',
        ];

        $this->controls['iconSize'] = [
            'label' => esc_html__('Icon Size', 'bricksforge'),
            'type'  => 'number',
            'units' => true,
            'css'   => [
                [
                    'property' => 'font-size',
                    'selector' => '.brf-calendar-options .brf-calendar-icon',
                ],
            ],
            'required' => ['calendarType', '=', 'multiple'],
            'group'    => 'dropdown',
        ];

        $this->controls['iconColor'] = [
            'label' => esc_html__('Icon Color', 'bricksforge'),
            'type'  => 'color',
            'css'   => [
                [
                    'property' => 'color',
                    'selector' => '.brf-calendar-options .brf-calendar-icon',
                ],
            ],
            'required' => ['calendarType', '=', 'multiple'],
            'group'    => 'dropdown',
        ];

        $this->controls['dropdownBorderRadius'] = [
            'label' => esc_html__('Dropdown Border Radius', 'bricksforge'),
            'type'  => 'number',
            'units' => true,
            'css'   => [
                [
                    'property' => 'border-radius',
                    'selector' => '.brf-calendar-options .brf-calendar-dropdown',
                ],
            ],
            'required' => ['calendarType', '=', 'multiple'],
            'group'    => 'dropdown',
        ];

        $this->controls['optionFocusBackground'] = [
            'label' => esc_html__('Option Focus Background', 'bricksforge'),
            'type'  => 'background',
            'css'   => [
                [
                    'property' => 'background',
                    'selector' => '.brf-calendar-options .brf-calendar-option:focus',
                ],
            ],
            'required' => ['calendarType', '=', 'multiple'],
            'group'    => 'dropdown',
        ];

        $this->controls['dropdownMaxHeight'] = [
            'label' => esc_html__('Dropdown Max Height', 'bricksforge'),
            'type'  => 'number',
            'units' => true,
            'css'   => [
                [
                    'property' => 'max-height',
                    'selector' => '.brf-calendar-options .brf-calendar-dropdown',
                ],
            ],
            'required' => ['calendarType', '=', 'multiple'],
            'group'    => 'dropdown',
        ];

        $this->controls['dropdownZIndex'] = [
            'label' => esc_html__('Dropdown Z-Index', 'bricksforge'),
            'type'  => 'number',
            'css'   => [
                [
                    'property' => 'z-index',
                    'selector' => '.brf-calendar-options .brf-calendar-dropdown',
                ],
            ],
            'default' => '1000',
            'required' => ['calendarType', '=', 'multiple'],
            'group'    => 'dropdown',
        ];

        // Icon Controls
        $this->controls['icon'] = [
            'label' => esc_html__('Icon', 'bricksforge'),
            'type'  => 'icon',
            'group' => 'icon',
        ];

        $this->controls['iconTypography'] = [
            'label'    => esc_html__('Typography', 'bricksforge'),
            'type'     => 'typography',
            'css'      => [
                [
                    'property' => 'font',
                    'selector' => 'i',
                ],
            ],
            'required' => ['icon.icon', '!=', ''],
            'group'    => 'icon',
        ];

        $this->controls['iconPosition'] = [
            'label'       => esc_html__('Position', 'bricksforge'),
            'type'        => 'select',
            'options'     => $this->control_options['iconPosition'],
            'inline'      => true,
            'placeholder' => esc_html__('Right', 'bricksforge'),
            'required'    => ['icon', '!=', ''],
            'group'       => 'icon',
        ];

        $this->controls['iconGap'] = [
            'label'    => esc_html__('Gap', 'bricksforge'),
            'type'     => 'number',
            'units'    => true,
            'css'      => [
                [
                    'property' => 'gap',
                ],
            ],
            'required' => ['icon', '!=', ''],
            'group'    => 'icon',
        ];

        $this->controls['iconSpace'] = [
            'label'    => esc_html__('Space between', 'bricksforge'),
            'type'     => 'checkbox',
            'css'      => [
                [
                    'property' => 'justify-content',
                    'value'    => 'space-between',
                ],
            ],
            'required' => ['icon', '!=', ''],
            'group'    => 'icon',
        ];

        // Dropdown Icon Controls
        $this->controls['googleIcon'] = [
            'label' => esc_html__('Google Calendar Icon', 'bricksforge'),
            'type'  => 'icon',
            'default' => [
                'library' => 'fontawesomeBrands',
                'icon' => 'fab fa-google',
            ],
            'required' => ['calendarType', '=', 'multiple'],
            'group'    => 'dropdownIcons',
        ];

        $this->controls['outlookIcon'] = [
            'label' => esc_html__('Outlook Icon', 'bricksforge'),
            'type'  => 'icon',
            'default' => [
                'library' => 'fontawesomeBrands',
                'icon' => 'fab fa-microsoft',
            ],
            'required' => ['calendarType', '=', 'multiple'],
            'group'    => 'dropdownIcons',
        ];

        $this->controls['appleIcon'] = [
            'label' => esc_html__('Apple Calendar Icon', 'bricksforge'),
            'type'  => 'icon',
            'default' => [
                'library' => 'fontawesomeBrands',
                'icon' => 'fab fa-apple',
            ],
            'required' => ['calendarType', '=', 'multiple'],
            'group'    => 'dropdownIcons',
        ];

        $this->controls['icsIcon'] = [
            'label' => esc_html__('ICS Download Icon', 'bricksforge'),
            'type'  => 'icon',
            'default' => [
                'library' => 'fontawesome',
                'icon' => 'fa fa-calendar',
            ],
            'required' => ['calendarType', '=', 'multiple'],
            'group'    => 'dropdownIcons',
        ];
    }

    private function get_timezone_options()
    {
        // Get all available timezones
        $timezone_identifiers = timezone_identifiers_list();
        $timezones = [];

        // Add UTC first
        $timezones['UTC'] = 'UTC';

        // Common timezones with user-friendly names
        $common_timezones = [
            // Europe
            'Europe/London' => 'Europe/London (GMT/BST)',
            'Europe/Berlin' => 'Europe/Berlin (CET/CEST)',
            'Europe/Paris' => 'Europe/Paris (CET/CEST)',
            'Europe/Rome' => 'Europe/Rome (CET/CEST)',
            'Europe/Madrid' => 'Europe/Madrid (CET/CEST)',
            'Europe/Amsterdam' => 'Europe/Amsterdam (CET/CEST)',
            'Europe/Vienna' => 'Europe/Vienna (CET/CEST)',
            'Europe/Zurich' => 'Europe/Zurich (CET/CEST)',
            'Europe/Stockholm' => 'Europe/Stockholm (CET/CEST)',
            'Europe/Oslo' => 'Europe/Oslo (CET/CEST)',
            'Europe/Copenhagen' => 'Europe/Copenhagen (CET/CEST)',
            'Europe/Helsinki' => 'Europe/Helsinki (EET/EEST)',
            'Europe/Warsaw' => 'Europe/Warsaw (CET/CEST)',
            'Europe/Prague' => 'Europe/Prague (CET/CEST)',
            'Europe/Budapest' => 'Europe/Budapest (CET/CEST)',
            'Europe/Brussels' => 'Europe/Brussels (CET/CEST)',
            'Europe/Dublin' => 'Europe/Dublin (GMT/IST)',
            'Europe/Lisbon' => 'Europe/Lisbon (WET/WEST)',
            'Europe/Athens' => 'Europe/Athens (EET/EEST)',
            'Europe/Moscow' => 'Europe/Moscow (MSK)',

            // North America
            'America/New_York' => 'America/New_York (EST/EDT)',
            'America/Chicago' => 'America/Chicago (CST/CDT)',
            'America/Denver' => 'America/Denver (MST/MDT)',
            'America/Los_Angeles' => 'America/Los_Angeles (PST/PDT)',
            'America/Toronto' => 'America/Toronto (EST/EDT)',
            'America/Vancouver' => 'America/Vancouver (PST/PDT)',
            'America/Montreal' => 'America/Montreal (EST/EDT)',
            'America/Phoenix' => 'America/Phoenix (MST)',
            'America/Anchorage' => 'America/Anchorage (AKST/AKDT)',
            'Pacific/Honolulu' => 'Pacific/Honolulu (HST)',

            // South America
            'America/Sao_Paulo' => 'America/Sao_Paulo (BRT/BRST)',
            'America/Argentina/Buenos_Aires' => 'America/Buenos_Aires (ART)',
            'America/Lima' => 'America/Lima (PET)',
            'America/Bogota' => 'America/Bogota (COT)',
            'America/Santiago' => 'America/Santiago (CLT/CLST)',

            // Asia
            'Asia/Tokyo' => 'Asia/Tokyo (JST)',
            'Asia/Shanghai' => 'Asia/Shanghai (CST)',
            'Asia/Hong_Kong' => 'Asia/Hong_Kong (HKT)',
            'Asia/Singapore' => 'Asia/Singapore (SGT)',
            'Asia/Seoul' => 'Asia/Seoul (KST)',
            'Asia/Kolkata' => 'Asia/Kolkata (IST)',
            'Asia/Dubai' => 'Asia/Dubai (GST)',
            'Asia/Bangkok' => 'Asia/Bangkok (ICT)',
            'Asia/Jakarta' => 'Asia/Jakarta (WIB)',
            'Asia/Manila' => 'Asia/Manila (PHT)',
            'Asia/Taipei' => 'Asia/Taipei (CST)',
            'Asia/Kuala_Lumpur' => 'Asia/Kuala_Lumpur (MYT)',
            'Asia/Istanbul' => 'Asia/Istanbul (TRT)',
            'Asia/Jerusalem' => 'Asia/Jerusalem (IST/IDT)',

            // Australia & Oceania
            'Australia/Sydney' => 'Australia/Sydney (AEST/AEDT)',
            'Australia/Melbourne' => 'Australia/Melbourne (AEST/AEDT)',
            'Australia/Brisbane' => 'Australia/Brisbane (AEST)',
            'Australia/Perth' => 'Australia/Perth (AWST)',
            'Australia/Adelaide' => 'Australia/Adelaide (ACST/ACDT)',
            'Pacific/Auckland' => 'Pacific/Auckland (NZST/NZDT)',

            // Africa
            'Africa/Cairo' => 'Africa/Cairo (EET/EEST)',
            'Africa/Johannesburg' => 'Africa/Johannesburg (SAST)',
            'Africa/Lagos' => 'Africa/Lagos (WAT)',
            'Africa/Nairobi' => 'Africa/Nairobi (EAT)',
            'Africa/Casablanca' => 'Africa/Casablanca (WET/WEST)',
        ];

        // Add common timezones
        foreach ($common_timezones as $tz => $label) {
            if (in_array($tz, $timezone_identifiers)) {
                $timezones[$tz] = $label;
            }
        }

        // Add separator
        $timezones['---'] = '--- All Timezones ---';

        // Add all other timezones
        foreach ($timezone_identifiers as $tz) {
            if (!isset($timezones[$tz]) && $tz !== '---') {
                $timezones[$tz] = $tz;
            }
        }

        return $timezones;
    }

    private function get_user_timezone()
    {
        // Try to get timezone from various sources
        $user_timezone = null;

        // 1. Check if user has a timezone preference stored
        if (is_user_logged_in()) {
            $user_timezone = get_user_meta(get_current_user_id(), 'timezone', true);
        }

        // 2. Try to detect from browser headers (Accept-Language)
        if (empty($user_timezone) && isset($_SERVER['HTTP_ACCEPT_LANGUAGE'])) {
            $user_timezone = $this->guess_timezone_from_locale($_SERVER['HTTP_ACCEPT_LANGUAGE']);
        }

        // 3. Fallback to WordPress timezone
        if (empty($user_timezone)) {
            $user_timezone = wp_timezone_string();
        }

        // Validate that the timezone exists in our options
        $available_timezones = array_keys($this->get_timezone_options());
        if (!in_array($user_timezone, $available_timezones)) {
            $user_timezone = wp_timezone_string();
        }

        return $user_timezone;
    }

    private function guess_timezone_from_locale($accept_language)
    {
        // Simple mapping of common locales to timezones
        $locale_timezone_map = [
            'de' => 'Europe/Berlin',
            'en-GB' => 'Europe/London',
            'en-US' => 'America/New_York',
            'fr' => 'Europe/Paris',
            'es' => 'Europe/Madrid',
            'it' => 'Europe/Rome',
            'nl' => 'Europe/Amsterdam',
            'pt' => 'Europe/Lisbon',
            'ru' => 'Europe/Moscow',
            'ja' => 'Asia/Tokyo',
            'zh' => 'Asia/Shanghai',
            'ko' => 'Asia/Seoul',
            'ar' => 'Asia/Dubai',
            'hi' => 'Asia/Kolkata',
            'th' => 'Asia/Bangkok',
            'vi' => 'Asia/Ho_Chi_Minh',
            'id' => 'Asia/Jakarta',
            'ms' => 'Asia/Kuala_Lumpur',
            'tl' => 'Asia/Manila',
            'tr' => 'Asia/Istanbul',
            'he' => 'Asia/Jerusalem',
            'sv' => 'Europe/Stockholm',
            'no' => 'Europe/Oslo',
            'da' => 'Europe/Copenhagen',
            'fi' => 'Europe/Helsinki',
            'pl' => 'Europe/Warsaw',
            'cs' => 'Europe/Prague',
            'hu' => 'Europe/Budapest',
            'el' => 'Europe/Athens',
            'pt-BR' => 'America/Sao_Paulo',
            'es-AR' => 'America/Argentina/Buenos_Aires',
            'es-CL' => 'America/Santiago',
            'es-CO' => 'America/Bogota',
            'es-PE' => 'America/Lima',
            'en-AU' => 'Australia/Sydney',
            'en-NZ' => 'Pacific/Auckland',
            'en-CA' => 'America/Toronto',
            'fr-CA' => 'America/Montreal',
        ];

        // Parse the Accept-Language header
        $languages = explode(',', $accept_language);
        foreach ($languages as $language) {
            $lang = trim(explode(';', $language)[0]);

            // Check exact match first
            if (isset($locale_timezone_map[$lang])) {
                return $locale_timezone_map[$lang];
            }

            // Check language code only (e.g., 'de' from 'de-DE')
            $lang_code = explode('-', $lang)[0];
            if (isset($locale_timezone_map[$lang_code])) {
                return $locale_timezone_map[$lang_code];
            }
        }

        return null;
    }

    public function get_nestable_children()
    {
        return [
            [
                'name'     => 'text-basic',
                'label'    => esc_html__('Button Text', 'bricksforge'),
                'settings' => [
                    'text' => esc_html__('Add To Calendar', 'bricksforge'),
                    'tag' => 'span',
                ]
            ],
        ];
    }

    public function render()
    {
        $settings = $this->settings;

        $this->set_attribute('_root', 'class', 'brf-add-to-calendar-button-wrapper');

        $this->set_attribute('_button', 'class', 'bricks-button');
        $this->set_attribute('_button', 'class', 'brf-add-to-calendar-button');

        if (isset($settings['cssClass']) && !empty($settings['cssClass'])) {
            $this->set_attribute('_button', 'class', bricks_render_dynamic_data($settings['cssClass']));
        }

        if (!empty($settings['size'])) {
            $this->set_attribute('_button', 'class', $settings['size']);
        }

        if (!empty($settings['style'])) {
            // Outline
            if (isset($settings['outline'])) {
                $this->set_attribute('_button', 'class', 'outline');
                $this->set_attribute('_button', 'class', "bricks-color-{$settings['style']}");
            }

            // Fill (= default)
            else {
                $this->set_attribute('_button', 'class', "bricks-background-{$settings['style']}");
            }
        }

        // Button circle
        if (isset($settings['circle'])) {
            $this->set_attribute('_button', 'class', 'circle');
        }

        if (isset($settings['block'])) {
            $this->set_attribute('_button', 'class', 'block');
        }

        // Calendar event data
        $calendar_data = $this->prepare_calendar_data($settings);

        // Add calendar data as data attributes
        foreach ($calendar_data as $key => $value) {
            if (is_array($value)) {
                // Convert arrays to JSON for data attributes
                $this->set_attribute('_button', 'data-' . $key, json_encode($value));
            } else {
                $this->set_attribute('_button', 'data-' . $key, $value);
            }
        }

        // Link handling for calendar
        if (!empty($settings['calendarType']) && $settings['calendarType'] !== 'ics') {
            $calendar_url = $this->generate_calendar_url($settings, $calendar_data);
            if ($calendar_url) {
                $this->tag = 'a';
                $this->set_attribute('_button', 'href', $calendar_url);
                $this->set_attribute('_button', 'target', '_blank');
                $this->set_attribute('_button', 'rel', 'noopener noreferrer');
            }
        } elseif (!empty($settings['link'])) {
            $this->tag = 'a';
            $this->set_link_attributes('_button', $settings['link']);
        }

        $output = "<div {$this->render_attributes('_root')}>";
        $output .= "<{$this->tag} {$this->render_attributes('_button')}>";
        $icon          = !empty($settings['icon']) ? self::render_icon($settings['icon']) : false;
        $icon_position = !empty($settings['iconPosition']) ? $settings['iconPosition'] : 'right';

        if ($icon && $icon_position === 'left') {
            $output .= $icon;
        }

        $output .= Frontend::render_children($this);

        if ($icon && $icon_position === 'right') {
            $output .= $icon;
        }

        $output .= "</{$this->tag}>";


        // Add multiple calendar options if selected
        if (!empty($settings['calendarType']) && $settings['calendarType'] === 'multiple') {
            $output .= $this->render_multiple_calendar_options($settings, $calendar_data);
        }

        $output .= "</div>";

        echo $output;
    }

    public function convert_element_settings_to_block($settings)
    {
        $text = isset($settings['text']) ? trim($settings['text']) : false;

        if (!$text) {
            return;
        }

        $text = str_replace('draggable="false"', '', $text);
        $text = str_replace('draggable="true"', '', $text);

        $block = ['blockName' => 'core/buttons'];

        $attributes = [];

        $html = '<div class="wp-block-buttons"><!-- wp:button -->';

        if (isset($settings['outline'])) {
            $attributes['className'] = 'is-style-outline';
            $html                   .= '<div class="wp-block-button is-style-outline">';
        } else {
            $html .= '<div class="wp-block-button"><a class="wp-block-button__link">';
        }

        if (isset($settings['_border']['radius']['top'])) {
            $attributes['borderRadius'] = intval($settings['_border']['radius']['top']);
        }

        $html .= $text . '</a></div><!-- /wp:button --></div>';

        $block['attrs']        = $attributes;
        $block['innerContent'] = [$html];

        return $block;
    }

    public function convert_block_to_element_settings($block, $attributes)
    {
        $text = strip_tags($block['innerHTML']);

        $element_settings = [
            'text'  => $text,
            'style' => 'dark',
        ];

        $border_radius = isset($attributes['borderRadius']) ? intval($attributes['borderRadius']) : false;

        // Outline
        if (isset($attributes['className']) && strpos($attributes['className'], 'is-style-outline') !== false) {
            $element_settings['outline'] = true;
        }

        if ($border_radius) {
            $element_settings['_border'] = [
                'radius' => [
                    'top'    => $border_radius,
                    'right'  => $border_radius,
                    'bottom' => $border_radius,
                    'left'   => $border_radius,
                ],
            ];
        }

        return $element_settings;
    }

    private function prepare_calendar_data($settings)
    {
        $data = [];

        // Get timezone - parse dynamic data
        $timezone = !empty($settings['timezone']) ? bricks_render_dynamic_data($settings['timezone']) : wp_timezone_string();
        $data['timezone'] = $timezone;

        // Calendar type and filename - parse dynamic data
        $data['calendar-type'] = !empty($settings['calendarType']) ? bricks_render_dynamic_data($settings['calendarType']) : 'ics';
        $data['filename'] = !empty($settings['filename']) ? bricks_render_dynamic_data($settings['filename']) : 'event';

        // Process events from repeater
        $data['events'] = [];
        if (!empty($settings['events'])) {
            foreach ($settings['events'] as $event) {
                $event_data = $this->prepare_single_event_data($event, $timezone);
                if ($event_data) {
                    $data['events'][] = $event_data;
                }
            }
        }

        // For backward compatibility and single event handling, use first event as main event
        if (!empty($data['events'])) {
            $main_event = $data['events'][0];
            $data['title'] = isset($main_event['title']) ? $main_event['title'] : '';
            $data['description'] = isset($main_event['description']) ? $main_event['description'] : '';
            $data['location'] = isset($main_event['location']) ? $main_event['location'] : '';
            $data['url'] = isset($main_event['url']) ? $main_event['url'] : '';
            $data['start'] = isset($main_event['start']) ? $main_event['start'] : '';
            $data['end'] = isset($main_event['end']) ? $main_event['end'] : '';
            $data['all-day'] = isset($main_event['all-day']) ? $main_event['all-day'] : '';
            if (!empty($main_event['reminder'])) {
                $data['reminder'] = $main_event['reminder'];
            }
            if (!empty($main_event['enableReminder'])) {
                $data['enableReminder'] = $main_event['enableReminder'];
            }
        }

        // Multiple events flag
        $data['multiple-events'] = count($data['events']) > 1 ? 'true' : 'false';

        return $data;
    }

    private function prepare_single_event_data($event_settings, $timezone)
    {
        $event_data = [];

        // Event details - parse dynamic data
        $event_data['title'] = !empty($event_settings['eventTitle']) ? bricks_render_dynamic_data($event_settings['eventTitle']) : 'Event';
        $event_data['description'] = !empty($event_settings['eventDescription']) ? bricks_render_dynamic_data($event_settings['eventDescription']) : '';
        $event_data['location'] = !empty($event_settings['eventLocation']) ? bricks_render_dynamic_data($event_settings['eventLocation']) : '';
        $event_data['url'] = !empty($event_settings['eventUrl']) ? bricks_render_dynamic_data($event_settings['eventUrl']) : '';

        // Date and time - timezone is already parsed in prepare_calendar_data
        $event_data['timezone'] = $timezone;
        $event_data['all-day'] = !empty($event_settings['allDay']) ? 'true' : 'false';

        // Check date method - dynamic or static (default)
        $date_method = !empty($event_settings['eventDateMethod']) ? $event_settings['eventDateMethod'] : 'static';

        if ($date_method === 'dynamic') {
            // Dynamic date method - use startDateDynamic and endDateDynamic fields
            if (!empty($event_settings['startDateDynamic'])) {
                $start_datetime = bricks_render_dynamic_data($event_settings['startDateDynamic']);

                if (!empty($event_settings['allDay'])) {
                    // For all-day events, extract only the date part
                    $event_data['start'] = date('Y-m-d', strtotime($start_datetime));
                } else {
                    // For timed events, use the full datetime
                    // Ensure proper format (add seconds if missing)
                    $parsed_start = date('Y-m-d\TH:i:s', strtotime($start_datetime));
                    $event_data['start'] = $parsed_start !== false ? $parsed_start : $start_datetime;
                }

                // Handle end date
                if (!empty($event_settings['endDateDynamic'])) {
                    $end_datetime = bricks_render_dynamic_data($event_settings['endDateDynamic']);

                    if (!empty($event_settings['allDay'])) {
                        // For all-day events, extract only the date part
                        $end_date_only = date('Y-m-d', strtotime($end_datetime));

                        // For ICS compatibility, all-day events end date should be the day AFTER the last day
                        // So if the event goes until 27.06, the ICS end date should be 28.06
                        $end_date_obj = new \DateTime($end_date_only);
                        $end_date_obj->add(new \DateInterval('P1D'));
                        $event_data['end'] = $end_date_obj->format('Y-m-d');
                    } else {
                        // For timed events, use the full datetime
                        $parsed_end = date('Y-m-d\TH:i:s', strtotime($end_datetime));
                        $event_data['end'] = $parsed_end !== false ? $parsed_end : $end_datetime;
                    }
                } else {
                    // No end date provided, set same as start or add 1 hour
                    if (!empty($event_settings['allDay'])) {
                        // For all-day event without end date, add 1 day to start for ICS compatibility
                        $start_date_obj = new \DateTime($event_data['start']);
                        $start_date_obj->add(new \DateInterval('P1D'));
                        $event_data['end'] = $start_date_obj->format('Y-m-d');
                    } else {
                        try {
                            $start_dt = new \DateTime($event_data['start']);
                            $start_dt->add(new \DateInterval('PT1H'));
                            $event_data['end'] = $start_dt->format('Y-m-d\TH:i:s');
                        } catch (\Exception $e) {
                            $event_data['end'] = $event_data['start'];
                        }
                    }
                }
            } else {
                // No dynamic start date provided, skip this event
                return null;
            }
        } else {
            // Static date method - use existing logic with separate date/time fields
            if (!empty($event_settings['startDate'])) {
                $start_date = bricks_render_dynamic_data($event_settings['startDate']);
                $start_time = !empty($event_settings['startTime']) ? bricks_render_dynamic_data($event_settings['startTime']) : '09:00';

                if (!empty($event_settings['allDay'])) {
                    // For all-day events, extract only the date part
                    $event_data['start'] = date('Y-m-d', strtotime($start_date));
                } else {
                    // For timed events, always include time
                    $start_date_only = date('Y-m-d', strtotime($start_date));

                    if (!empty($event_settings['startTime']) && strlen($start_time) === 5 && strpos($start_time, ':') === 2) {
                        // Use specified start time
                        $event_data['start'] = $start_date_only . 'T' . $start_time . ':00';
                    } else {
                        // Use default start time (09:00) for timed events
                        $event_data['start'] = $start_date_only . 'T09:00:00';
                    }
                }

                // End date/time - parse dynamic data
                if (!empty($event_settings['endDate'])) {
                    $end_date = bricks_render_dynamic_data($event_settings['endDate']);
                    $end_time = !empty($event_settings['endTime']) ? bricks_render_dynamic_data($event_settings['endTime']) : '17:00';

                    // Check if end date is the same as start date (user didn't specify different end date)
                    $start_date_only = date('Y-m-d', strtotime($start_date));
                    $end_date_only = date('Y-m-d', strtotime($end_date));

                    if (!empty($event_settings['allDay'])) {
                        if ($start_date_only === $end_date_only) {
                            // Same day all-day event - add 1 day for ICS compatibility
                            $end_datetime = new \DateTime($end_date_only);
                            $end_datetime->add(new \DateInterval('P1D'));
                            $event_data['end'] = $end_datetime->format('Y-m-d');
                        }
                    } else {
                        // For timed events, always include time
                        if (!empty($event_settings['endTime']) && strlen($end_time) === 5 && strpos($end_time, ':') === 2) {
                            // Use specified end time
                            $event_data['end'] = $end_date_only . 'T' . $end_time . ':00';
                        } else {
                            // Use default end time (17:00) for timed events
                            $event_data['end'] = $end_date_only . 'T17:00:00';
                        }
                    }
                } else {
                    // If no end date, but we have start date
                    if (!empty($event_settings['allDay'])) {
                        $event_data['end'] = $event_data['start'];
                    } else {
                        // Check if we have an end time specified
                        $end_time = !empty($event_settings['endTime']) ? bricks_render_dynamic_data($event_settings['endTime']) : '';

                        if (!empty($end_time) && strlen($end_time) === 5 && strpos($end_time, ':') === 2) {
                            // Use start date with end time
                            $start_date = bricks_render_dynamic_data($event_settings['startDate']);

                            // Extract just the date part from start_date
                            if (strpos($start_date, ' ') !== false || strpos($start_date, 'T') !== false) {
                                $date_part = date('Y-m-d', strtotime($start_date));
                            } else {
                                $date_part = $start_date;
                            }

                            $event_data['end'] = $date_part . 'T' . $end_time . ':00';
                        } else {
                            // Fallback: use start time + 1 hour
                            try {
                                $start_datetime = new \DateTime($event_data['start']);
                                $start_datetime->add(new \DateInterval('PT1H'));
                                $event_data['end'] = $start_datetime->format('Y-m-d\TH:i:s');
                            } catch (\Exception $e) {
                                // Fallback if parsing fails
                                $event_data['end'] = $event_data['start'];
                            }
                        }
                    }
                }
            } else {
                // No start date provided, skip this event
                return null;
            }
        }

        // Reminder - parse dynamic data for reminderMinutes
        if (!empty($event_settings['enableReminder'])) {
            $event_data['enableReminder'] = true;
            $reminder_minutes = !empty($event_settings['reminderMinutes']) ? bricks_render_dynamic_data($event_settings['reminderMinutes']) : '15';
            // Ensure it's a valid number
            $event_data['reminder'] = is_numeric($reminder_minutes) ? $reminder_minutes : '15';
        } else {
            $event_data['enableReminder'] = false;
        }

        return $event_data;
    }

    private function generate_calendar_url($settings, $calendar_data)
    {
        $calendar_type = !empty($settings['calendarType']) ? $settings['calendarType'] : 'ics';

        switch ($calendar_type) {
            case 'google':
                return $this->generate_google_calendar_url($calendar_data);
            case 'outlook':
                return $this->generate_outlook_calendar_url($calendar_data);
            case 'apple':
                // Apple Calendar uses the same ICS format, but we'll handle it via JavaScript
                return '#';
            default:
                return '#';
        }
    }

    private function generate_google_calendar_url($data)
    {
        $base_url = 'https://calendar.google.com/calendar/render?action=TEMPLATE';

        // Prepare description with URL if provided
        $description = $data['description'];
        if (!empty($data['url'])) {
            if (!empty($description)) {
                $description .= "\n\n---\n\n";
            }
            $description .= " 🔗 URL: " . $data['url'];
        }

        $params = [
            'text' => $data['title'],
            'details' => $description,
            'location' => $data['location'],
        ];

        // Format dates for Google Calendar
        if (!empty($data['start'])) {
            $start_date = $this->format_date_for_google($data['start'], $data['timezone'], $data['all-day'] === 'true');
            $end_date = !empty($data['end']) ? $this->format_date_for_google($data['end'], $data['timezone'], $data['all-day'] === 'true') : $start_date;

            $params['dates'] = $start_date . '/' . $end_date;
        }

        // http_build_query() handles URL encoding automatically
        $query_string = http_build_query($params);
        return $base_url . '&' . $query_string;
    }

    private function generate_outlook_calendar_url($data)
    {
        $base_url = 'https://outlook.live.com/calendar/0/deeplink/compose';

        // Prepare body with URL if provided
        $body = $data['description'];
        if (!empty($data['url'])) {
            if (!empty($body)) {
                $body .= "\n\n---\n\n";
            }
            $body .= " 🔗 URL: " . $data['url'];
        }

        $params = [
            'subject' => $data['title'],
            'body' => $body,
            'location' => $data['location'],
        ];

        // Format dates for Outlook
        if (!empty($data['start'])) {
            $start_date = $this->format_date_for_outlook($data['start'], $data['timezone'], $data['all-day'] === 'true');
            $end_date = !empty($data['end']) ? $this->format_date_for_outlook($data['end'], $data['timezone'], $data['all-day'] === 'true') : $start_date;

            $params['startdt'] = $start_date;
            $params['enddt'] = $end_date;
        }

        if ($data['all-day'] === 'true') {
            $params['allday'] = 'true';
        }

        // http_build_query() handles URL encoding automatically
        $query_string = http_build_query($params);
        return $base_url . '?' . $query_string;
    }

    private function format_date_for_google($date, $timezone, $all_day = false)
    {
        if ($all_day) {
            return date('Ymd', strtotime($date));
        } else {
            try {
                // Parse date components manually to avoid timezone conversions
                if (preg_match('/(\d{4})-(\d{2})-(\d{2})[T\s](\d{2}):(\d{2}):(\d{2})/', $date, $matches)) {
                    $year = (int)$matches[1];
                    $month = (int)$matches[2];
                    $day = (int)$matches[3];
                    $hour = (int)$matches[4];
                    $minute = (int)$matches[5];
                    $second = (int)$matches[6];

                    // Create datetime in the specified timezone
                    $datetime = new \DateTime();
                    $datetime->setTimezone(new \DateTimeZone($timezone));
                    $datetime->setDate($year, $month, $day);
                    $datetime->setTime($hour, $minute, $second);

                    // Convert to UTC for Google Calendar
                    $datetime->setTimezone(new \DateTimeZone('UTC'));
                    return $datetime->format('Ymd\THis\Z');
                }

                // Fallback if regex doesn't match
                $datetime = new \DateTime($date);
                return $datetime->format('Ymd\THis\Z');
            } catch (\Exception $e) {
                // Fallback if all parsing fails
                $datetime = new \DateTime($date);
                return $datetime->format('Ymd\THis\Z');
            }
        }
    }

    private function format_date_for_outlook($date, $timezone, $all_day = false)
    {
        if ($all_day) {
            return date('Y-m-d', strtotime($date));
        } else {
            // Parse datetime without timezone first to avoid unwanted conversions
            // Outlook uses local time with timezone specified in URL parameters
            $datetime = new \DateTime($date);
            return $datetime->format('Y-m-d\TH:i:s');
        }
    }

    private function render_multiple_calendar_options($settings, $calendar_data)
    {
        $output = '<div class="brf-calendar-options">';
        $output .= '<div class="brf-calendar-dropdown">';

        // Google Calendar
        $google_url = $this->generate_google_calendar_url($calendar_data);
        $output .= '<a href="' . esc_url($google_url) . '" target="_blank" rel="noopener noreferrer" class="brf-calendar-option" data-type="google">';

        // Google Icon
        if (!empty($settings['googleIcon'])) {
            $output .= '<span class="brf-calendar-icon">';
            $output .= self::render_icon($settings['googleIcon']);
            $output .= '</span>';
        } else {
            $output .= '<span class="brf-calendar-icon">📅</span>';
        }

        $output .= '<span>' . esc_html__('Google Calendar', 'bricksforge') . '</span>';
        $output .= '</a>';

        // Outlook
        $outlook_url = $this->generate_outlook_calendar_url($calendar_data);
        $output .= '<a href="' . esc_url($outlook_url) . '" target="_blank" rel="noopener noreferrer" class="brf-calendar-option" data-type="outlook">';

        // Outlook Icon
        if (!empty($settings['outlookIcon'])) {
            $output .= '<span class="brf-calendar-icon">';
            $output .= self::render_icon($settings['outlookIcon']);
            $output .= '</span>';
        } else {
            $output .= '<span class="brf-calendar-icon">📧</span>';
        }

        $output .= '<span>' . esc_html__('Outlook', 'bricksforge') . '</span>';
        $output .= '</a>';

        // Apple Calendar / ICS Download
        $output .= '<a href="#" class="brf-calendar-option" data-type="apple">';

        // Apple Icon
        if (!empty($settings['appleIcon'])) {
            $output .= '<span class="brf-calendar-icon">';
            $output .= self::render_icon($settings['appleIcon']);
            $output .= '</span>';
        } else {
            $output .= '<span class="brf-calendar-icon">🍎</span>';
        }

        $output .= '<span>' . esc_html__('Apple Calendar', 'bricksforge') . '</span>';
        $output .= '</a>';

        // ICS Download
        $output .= '<a href="#" class="brf-calendar-option" data-type="ics">';

        // ICS Icon
        if (!empty($settings['icsIcon'])) {
            $output .= '<span class="brf-calendar-icon">';
            $output .= self::render_icon($settings['icsIcon']);
            $output .= '</span>';
        } else {
            $output .= '<span class="brf-calendar-icon">📄</span>';
        }

        $output .= '<span>' . esc_html__('Download ICS', 'bricksforge') . '</span>';
        $output .= '</a>';

        $output .= '</div>';
        $output .= '</div>';

        return $output;
    }

    public function generate_ics_content($data)
    {
        $ics_content = "BEGIN:VCALENDAR\r\n";
        $ics_content .= "VERSION:2.0\r\n";
        $ics_content .= "PRODID:-//Bricksforge//Add to Calendar Button//EN\r\n";
        $ics_content .= "CALSCALE:GREGORIAN\r\n";
        $ics_content .= "METHOD:PUBLISH\r\n";

        // Add timezone information for better compatibility
        $timezone = !empty($data['timezone']) ? $data['timezone'] : wp_timezone_string();
        if ($timezone !== 'UTC') {
            $ics_content .= "X-WR-TIMEZONE:" . $timezone . "\r\n";

            // Add VTIMEZONE block for proper timezone definition
            $vtimezone = $this->generate_vtimezone_block($timezone);
            if ($vtimezone) {
                $ics_content .= $vtimezone;
            }
        }

        // Add all events from the events array
        if (!empty($data['events'])) {
            foreach ($data['events'] as $event) {
                $ics_content .= $this->generate_single_event_ics($event);
            }
        } else {
            // Fallback: use main event data for backward compatibility
            $ics_content .= $this->generate_single_event_ics($data);
        }

        $ics_content .= "END:VCALENDAR\r\n";

        return $ics_content;
    }

    private function generate_single_event_ics($data)
    {
        $ics_content = "BEGIN:VEVENT\r\n";

        // Generate unique ID
        $uid = uniqid() . '@' . (!empty($_SERVER['HTTP_HOST']) ? $_SERVER['HTTP_HOST'] : 'localhost');
        $ics_content .= "UID:" . $uid . "\r\n";

        // Current timestamp
        $dtstamp = gmdate('Ymd\THis\Z');
        $ics_content .= "DTSTAMP:" . $dtstamp . "\r\n";
        $ics_content .= "CREATED:" . $dtstamp . "\r\n";
        $ics_content .= "LAST-MODIFIED:" . $dtstamp . "\r\n";

        // Event details
        $ics_content .= "SUMMARY:" . $this->escape_ics_text($data['title']) . "\r\n";

        if (!empty($data['description'])) {
            $ics_content .= "DESCRIPTION:" . $this->escape_ics_text($data['description']) . "\r\n";
        }

        if (!empty($data['location'])) {
            $ics_content .= "LOCATION:" . $this->escape_ics_text($data['location']) . "\r\n";
        }

        if (!empty($data['url'])) {
            $ics_content .= "URL:" . $this->escape_ics_text($data['url']) . "\r\n";
        }

        // Status and transparency
        $ics_content .= "STATUS:CONFIRMED\r\n";
        $ics_content .= "TRANSP:OPAQUE\r\n";

        $timezone = !empty($data['timezone']) ? $data['timezone'] : wp_timezone_string();

        // Date and time with proper timezone handling
        if (!empty($data['start'])) {
            if ($data['all-day'] === 'true') {
                // All-day events use DATE format without timezone
                $start_date = date('Ymd', strtotime($data['start']));
                $ics_content .= "DTSTART;VALUE=DATE:" . $start_date . "\r\n";

                if (!empty($data['end'])) {
                    $end_date = date('Ymd', strtotime($data['end']));
                    $ics_content .= "DTEND;VALUE=DATE:" . $end_date . "\r\n";
                } else {
                    // For all-day events without end date, use next day
                    $end_datetime = new \DateTime($data['start']);
                    $end_datetime->add(new \DateInterval('P1D'));
                    $ics_content .= "DTEND;VALUE=DATE:" . $end_datetime->format('Ymd') . "\r\n";
                }
            } else {
                // Timed events with timezone support
                try {
                    // Parse datetime without timezone first to avoid unwanted conversions
                    $start_datetime = new \DateTime($data['start']);

                    if ($timezone === 'UTC') {
                        // For UTC, use Z suffix
                        $ics_content .= "DTSTART:" . $start_datetime->format('Ymd\THis\Z') . "\r\n";
                    } else {
                        // For other timezones, use TZID parameter with the original local time
                        // The time is treated as already being in the specified timezone
                        // For other timezones, use TZID parameter with local time
                        $ics_content .= "DTSTART;TZID=" . $timezone . ":" . $start_datetime->format('Ymd\THis') . "\r\n";
                    }

                    if (!empty($data['end'])) {
                        $end_datetime = new \DateTime($data['end'], new \DateTimeZone($timezone));

                        if ($timezone === 'UTC') {
                            $ics_content .= "DTEND:" . $end_datetime->format('Ymd\THis\Z') . "\r\n";
                        } else {
                            $ics_content .= "DTEND;TZID=" . $timezone . ":" . $end_datetime->format('Ymd\THis') . "\r\n";
                        }
                    } else {
                        // Default to 1 hour duration if no end time
                        $end_datetime = clone $start_datetime;
                        $end_datetime->add(new \DateInterval('PT1H'));

                        if ($timezone === 'UTC') {
                            $ics_content .= "DTEND:" . $end_datetime->format('Ymd\THis\Z') . "\r\n";
                        } else {
                            $ics_content .= "DTEND;TZID=" . $timezone . ":" . $end_datetime->format('Ymd\THis') . "\r\n";
                        }
                    }
                } catch (\Exception $e) {
                    // Fallback if datetime parsing fails - use UTC
                    $fallback_start = gmdate('Ymd\THis\Z');
                    $ics_content .= "DTSTART:" . $fallback_start . "\r\n";
                    $fallback_end = gmdate('Ymd\THis\Z', strtotime('+1 hour'));
                    $ics_content .= "DTEND:" . $fallback_end . "\r\n";
                }
            }
        } else {
            // Fallback if no start date provided - use current time in UTC
            $fallback_start = gmdate('Ymd\THis\Z');
            $ics_content .= "DTSTART:" . $fallback_start . "\r\n";
            $fallback_end = gmdate('Ymd\THis\Z', strtotime('+1 hour'));
            $ics_content .= "DTEND:" . $fallback_end . "\r\n";
        }

        // Reminder/Alarm
        if (!empty($data['reminder']) && !empty($data['enableReminder'])) {
            $ics_content .= "BEGIN:VALARM\r\n";
            $ics_content .= "TRIGGER:-PT" . intval($data['reminder']) . "M\r\n";
            $ics_content .= "ACTION:DISPLAY\r\n";
            $ics_content .= "DESCRIPTION:Reminder\r\n";
            $ics_content .= "END:VALARM\r\n";
        }

        $ics_content .= "END:VEVENT\r\n";

        return $ics_content;
    }

    /**
     * Generate VTIMEZONE block for common timezones
     */
    private function generate_vtimezone_block($timezone)
    {
        // Common timezone definitions
        $vtimezone_definitions = [
            'Europe/Berlin' => "BEGIN:VTIMEZONE\r\nTZID:Europe/Berlin\r\nBEGIN:DAYLIGHT\r\nTZOFFSETFROM:+0100\r\nTZOFFSETTO:+0200\r\nTZNAME:CEST\r\nDTSTART:19700329T020000\r\nRRULE:FREQ=YEARLY;BYMONTH=3;BYDAY=-1SU\r\nEND:DAYLIGHT\r\nBEGIN:STANDARD\r\nTZOFFSETFROM:+0200\r\nTZOFFSETTO:+0100\r\nTZNAME:CET\r\nDTSTART:19701025T030000\r\nRRULE:FREQ=YEARLY;BYMONTH=10;BYDAY=-1SU\r\nEND:STANDARD\r\nEND:VTIMEZONE\r\n",

            'Europe/London' => "BEGIN:VTIMEZONE\r\nTZID:Europe/London\r\nBEGIN:DAYLIGHT\r\nTZOFFSETFROM:+0000\r\nTZOFFSETTO:+0100\r\nTZNAME:BST\r\nDTSTART:19700329T010000\r\nRRULE:FREQ=YEARLY;BYMONTH=3;BYDAY=-1SU\r\nEND:DAYLIGHT\r\nBEGIN:STANDARD\r\nTZOFFSETFROM:+0100\r\nTZOFFSETTO:+0000\r\nTZNAME:GMT\r\nDTSTART:19701025T020000\r\nRRULE:FREQ=YEARLY;BYMONTH=10;BYDAY=-1SU\r\nEND:STANDARD\r\nEND:VTIMEZONE\r\n",

            'America/New_York' => "BEGIN:VTIMEZONE\r\nTZID:America/New_York\r\nBEGIN:DAYLIGHT\r\nTZOFFSETFROM:-0500\r\nTZOFFSETTO:-0400\r\nTZNAME:EDT\r\nDTSTART:19700308T020000\r\nRRULE:FREQ=YEARLY;BYMONTH=3;BYDAY=2SU\r\nEND:DAYLIGHT\r\nBEGIN:STANDARD\r\nTZOFFSETFROM:-0400\r\nTZOFFSETTO:-0500\r\nTZNAME:EST\r\nDTSTART:19701101T020000\r\nRRULE:FREQ=YEARLY;BYMONTH=11;BYDAY=1SU\r\nEND:STANDARD\r\nEND:VTIMEZONE\r\n",

            'America/Los_Angeles' => "BEGIN:VTIMEZONE\r\nTZID:America/Los_Angeles\r\nBEGIN:DAYLIGHT\r\nTZOFFSETFROM:-0800\r\nTZOFFSETTO:-0700\r\nTZNAME:PDT\r\nDTSTART:19700308T020000\r\nRRULE:FREQ=YEARLY;BYMONTH=3;BYDAY=2SU\r\nEND:DAYLIGHT\r\nBEGIN:STANDARD\r\nTZOFFSETFROM:-0700\r\nTZOFFSETTO:-0800\r\nTZNAME:PST\r\nDTSTART:19701101T020000\r\nRRULE:FREQ=YEARLY;BYMONTH=11;BYDAY=1SU\r\nEND:STANDARD\r\nEND:VTIMEZONE\r\n",

            'Europe/Paris' => "BEGIN:VTIMEZONE\r\nTZID:Europe/Paris\r\nBEGIN:DAYLIGHT\r\nTZOFFSETFROM:+0100\r\nTZOFFSETTO:+0200\r\nTZNAME:CEST\r\nDTSTART:19700329T020000\r\nRRULE:FREQ=YEARLY;BYMONTH=3;BYDAY=-1SU\r\nEND:DAYLIGHT\r\nBEGIN:STANDARD\r\nTZOFFSETFROM:+0200\r\nTZOFFSETTO:+0100\r\nTZNAME:CET\r\nDTSTART:19701025T030000\r\nRRULE:FREQ=YEARLY;BYMONTH=10;BYDAY=-1SU\r\nEND:STANDARD\r\nEND:VTIMEZONE\r\n",

            'Asia/Tokyo' => "BEGIN:VTIMEZONE\r\nTZID:Asia/Tokyo\r\nBEGIN:STANDARD\r\nTZOFFSETFROM:+0900\r\nTZOFFSETTO:+0900\r\nTZNAME:JST\r\nDTSTART:19700101T000000\r\nEND:STANDARD\r\nEND:VTIMEZONE\r\n",

            'Australia/Sydney' => "BEGIN:VTIMEZONE\r\nTZID:Australia/Sydney\r\nBEGIN:STANDARD\r\nTZOFFSETFROM:+1100\r\nTZOFFSETTO:+1000\r\nTZNAME:AEST\r\nDTSTART:19700405T030000\r\nRRULE:FREQ=YEARLY;BYMONTH=4;BYDAY=1SU\r\nEND:STANDARD\r\nBEGIN:DAYLIGHT\r\nTZOFFSETFROM:+1000\r\nTZOFFSETTO:+1100\r\nTZNAME:AEDT\r\nDTSTART:19701004T020000\r\nRRULE:FREQ=YEARLY;BYMONTH=10;BYDAY=1SU\r\nEND:DAYLIGHT\r\nEND:VTIMEZONE\r\n"
        ];

        // Also support common European timezones with same rules as their main cities
        $timezone_aliases = [
            'Europe/Vienna' => 'Europe/Berlin',
            'Europe/Zurich' => 'Europe/Berlin',
            'Europe/Amsterdam' => 'Europe/Berlin',
            'Europe/Rome' => 'Europe/Berlin',
            'Europe/Madrid' => 'Europe/Berlin',
            'Europe/Stockholm' => 'Europe/Berlin',
            'Europe/Oslo' => 'Europe/Berlin',
            'Europe/Copenhagen' => 'Europe/Berlin',
            'Europe/Warsaw' => 'Europe/Berlin',
            'Europe/Prague' => 'Europe/Berlin',
            'Europe/Budapest' => 'Europe/Berlin',
            'Europe/Brussels' => 'Europe/Berlin',
        ];

        // Check if we have a direct definition
        if (isset($vtimezone_definitions[$timezone])) {
            return $vtimezone_definitions[$timezone];
        }

        // Check if we have an alias
        if (isset($timezone_aliases[$timezone])) {
            $base_definition = $vtimezone_definitions[$timezone_aliases[$timezone]];
            // Replace the TZID with the actual timezone
            return str_replace('TZID:' . $timezone_aliases[$timezone], 'TZID:' . $timezone, $base_definition);
        }

        // For other timezones, try to generate a basic VTIMEZONE block
        try {
            $dt = new \DateTime('now', new \DateTimeZone($timezone));
            $offset = $dt->format('P'); // e.g., +02:00
            $offset_formatted = str_replace(':', '', $offset); // e.g., +0200

            return "BEGIN:VTIMEZONE\r\nTZID:" . $timezone . "\r\nBEGIN:STANDARD\r\nTZOFFSETFROM:" . $offset_formatted . "\r\nTZOFFSETTO:" . $offset_formatted . "\r\nTZNAME:" . $dt->format('T') . "\r\nDTSTART:19700101T000000\r\nEND:STANDARD\r\nEND:VTIMEZONE\r\n";
        } catch (\Exception $e) {
            // If all else fails, return empty (will fallback to UTC)
            return '';
        }
    }

    private function escape_ics_text($text)
    {
        // Escape special characters for ICS format
        $text = str_replace(['\\', ',', ';', "\n", "\r"], ['\\\\', '\\,', '\\;', '\\n', ''], $text);
        return $text;
    }

    public static function init_ajax_handlers()
    {
        add_action('wp_ajax_brf_generate_ics', [__CLASS__, 'ajax_generate_ics']);
        add_action('wp_ajax_nopriv_brf_generate_ics', [__CLASS__, 'ajax_generate_ics']);
    }

    public static function ajax_generate_ics()
    {
        // Verify nonce for security
        if (!wp_verify_nonce($_POST['nonce'], 'brf_calendar_nonce')) {
            wp_die('Security check failed');
        }

        $data = $_POST['calendar_data'];

        // Sanitize main data
        $sanitized_data = [
            'title' => sanitize_text_field($data['title']),
            'description' => sanitize_textarea_field($data['description']),
            'location' => sanitize_text_field($data['location']),
            'url' => esc_url_raw($data['url']),
            'start' => sanitize_text_field($data['start']),
            'end' => sanitize_text_field($data['end']),
            'timezone' => sanitize_text_field($data['timezone']),
            'all-day' => sanitize_text_field($data['all-day']),
            'reminder' => sanitize_text_field($data['reminder']),
            'enableReminder' => !empty($data['enableReminder']) ? $data['enableReminder'] : false,
            'filename' => sanitize_file_name($data['filename'])
        ];

        // Handle events array if present (for multiple events)
        if (!empty($data['events']) && is_array($data['events'])) {
            $sanitized_data['events'] = [];
            foreach ($data['events'] as $event) {
                $sanitized_event = [
                    'title' => sanitize_text_field($event['title']),
                    'description' => sanitize_textarea_field($event['description']),
                    'location' => sanitize_text_field($event['location']),
                    'url' => esc_url_raw($event['url']),
                    'start' => sanitize_text_field($event['start']),
                    'end' => sanitize_text_field($event['end']),
                    'timezone' => sanitize_text_field($event['timezone']),
                    'all-day' => sanitize_text_field($event['all-day']),
                    'reminder' => sanitize_text_field($event['reminder']),
                    'enableReminder' => !empty($event['enableReminder']) ? $event['enableReminder'] : false,
                ];
                $sanitized_data['events'][] = $sanitized_event;
            }
        }

        $instance = new self();
        $ics_content = $instance->generate_ics_content($sanitized_data);

        $filename = !empty($sanitized_data['filename']) ? $sanitized_data['filename'] : 'event';

        header('Content-Type: text/calendar; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '.ics"');
        header('Content-Length: ' . strlen($ics_content));

        echo $ics_content;
        wp_die();
    }
}

// Initialize AJAX handlers
add_action('init', ['Bricks\Brf_Add_To_Calendar_Button', 'init_ajax_handlers']);
