<?php
/**
 * Script to fix data-attributes in all PHP element files
 * Moves data-* attributes from set_attribute() calls to inline HTML
 */

$elementsDir = __DIR__ . '/elements/';
$processedFiles = [];
$errorFiles = [];

// List of files already processed
$alreadyProcessed = [
    'bc_offcanvas_menu.php',
    'bc_glowingcard.php',
    'next_overlay_shadows.php',
    'bc_flipbox.php',
    'bc_counter.php',
    'bc_circular_title.php',
    'bc_beforeafter_image.php',
    'bc_marquee.php',
    'bc_expanding_menu.php',
    'bc_typed.php',
    'bc_divider.php',
    'next_countdown.php',
    'bc_generatorism.php',
    'next_variable_font_hover.php',
    'bc_scribble_child.php',
    'next_arrow_button_v3.php',
    'bc_core_burguer.php',
    'next_arrow_button_v2.php',
    'next_cookies.php',
    'bc_clickandcopy.php',
    'next_circular_menu.php',
    'bc_3d_swap_title.php',
    'next_inverted_corners.php'
];

// Get all PHP files in elements directory
$files = glob($elementsDir . '*.php');

foreach ($files as $file) {
    $filename = basename($file);
    
    // Skip already processed files
    if (in_array($filename, $alreadyProcessed)) {
        continue;
    }
    
    $content = file_get_contents($file);
    $originalContent = $content;
    
    // Check if file has set_attribute with data-
    if (!preg_match('/\$this->set_attribute\s*\([^,]+,\s*[\'"]data-/', $content)) {
        continue;
    }
    
    // Find all set_attribute calls with data- attributes
    preg_match_all('/\$this->set_attribute\s*\(\s*([^,]+),\s*[\'"]data-([^"\']+)[\'"]\s*,\s*([^)]+)\s*\);?\s*\n?/m', $content, $matches, PREG_SET_ORDER);
    
    if (empty($matches)) {
        continue;
    }
    
    // Group data attributes by element
    $dataAttributes = [];
    foreach ($matches as $match) {
        $element = trim($match[1], " '\"");
        $attrName = 'data-' . $match[2];
        $attrValue = trim($match[3]);
        
        if (!isset($dataAttributes[$element])) {
            $dataAttributes[$element] = [];
        }
        $dataAttributes[$element][] = ['name' => $attrName, 'value' => $attrValue];
        
        // Remove the set_attribute line
        $content = str_replace($match[0], '', $content);
    }
    
    // Now find render_attributes calls and add data attributes inline
    foreach ($dataAttributes as $element => $attrs) {
        // Find patterns like: $output = "<tag {$this->render_attributes( 'element' )}>";
        $patterns = [
            // Pattern 1: Simple tag with render_attributes
            '/(<[^>]+\{\$this->render_attributes\s*\(\s*[\'"]' . preg_quote($element) . '[\'"]\s*\)\s*\})>/m',
            // Pattern 2: Tag with render_attributes and existing content
            '/(<[^>]+\{\$this->render_attributes\s*\(\s*[\'"]' . preg_quote($element) . '[\'"]\s*\)\s*\})\s*([^>]*>)/m',
            // Pattern 3: Self-closing tag
            '/(<[^>]+\{\$this->render_attributes\s*\(\s*[\'"]' . preg_quote($element) . '[\'"]\s*\)\s*\})\s*\/>/m',
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $content)) {
                // Build the data attributes string
                $dataAttrsStr = '';
                foreach ($attrs as $attr) {
                    $value = $attr['value'];
                    // Handle different value formats
                    if (strpos($value, '$') === 0) {
                        // Variable
                        $dataAttrsStr .= "\n    " . $attr['name'] . "='{{$value}}'";
                    } elseif (strpos($value, '"') !== false) {
                        // Already quoted
                        $dataAttrsStr .= "\n    " . $attr['name'] . "={$value}";
                    } else {
                        // String value
                        $value = trim($value, "'\"");
                        $dataAttrsStr .= "\n    " . $attr['name'] . "='{$value}'";
                    }
                }
                
                // Replace based on pattern type
                if (preg_match('/\/>\s*$/m', $pattern)) {
                    // Self-closing tag
                    $content = preg_replace($pattern, '$1' . $dataAttrsStr . ' />', $content);
                } else {
                    // Regular tag
                    $content = preg_replace($pattern, '$1' . $dataAttrsStr . '$2', $content);
                }
                break;
            }
        }
    }
    
    // Only write if content changed
    if ($content !== $originalContent) {
        if (file_put_contents($file, $content)) {
            $processedFiles[] = $filename;
            echo "✅ Processed: $filename\n";
        } else {
            $errorFiles[] = $filename;
            echo "❌ Error writing: $filename\n";
        }
    }
}

echo "\n=== SUMMARY ===\n";
echo "Files processed successfully: " . count($processedFiles) . "\n";
echo "Files with errors: " . count($errorFiles) . "\n";
echo "Files already processed: " . count($alreadyProcessed) . "\n";

if (!empty($processedFiles)) {
    echo "\nProcessed files:\n";
    foreach ($processedFiles as $file) {
        echo "  - $file\n";
    }
}

if (!empty($errorFiles)) {
    echo "\nFiles with errors:\n";
    foreach ($errorFiles as $file) {
        echo "  - $file\n";
    }
}