<?php

namespace SureCart\NextBricksLicensing;

/**
 * The settings class.
 */
class Settings {
	/**
	 * SureCart\NextBricksLicensing\Client
	 *
	 * @var object
	 */
	protected $client;

	/**
	 * Holds the option key
	 *
	 * @var string
	 */
	private $option_key;

	/**
	 * Holds the option name
	 *
	 * @var string
	 */
	private $name;

	/**
	 * Holds the menu arguments
	 *
	 * @var array
	 */
	private $menu_args;

	/**
	 * Create the pages.
	 *
	 * @param SureCart\NextBricksLicensing\Client $client The client.
	 */
	public function __construct( Client $client ) {
		$this->client     = $client;
		$this->name       = strtolower( preg_replace( '/\s+/', '', $this->client->name ) );
		$this->option_key = $this->name . '_license_options';
	}

	/**
	 * Add the settings page.
	 *
	 * @param array $args Settings page args.
	 *
	 * @return void
	 */
	public function add_page( $args ) {
		// store menu args for proper menu creation.
		$this->menu_args = wp_parse_args(
			$args,
			array(
				'type'               => 'menu', // Can be: menu, options, submenu.
				'page_title'         => 'Manage License',
				'menu_title'         => 'Manage License',
				'capability'         => 'manage_options',
				'menu_slug'          => $this->client->slug . '-manage-license',
				'icon_url'           => '',
				'position'           => null,
				'activated_redirect' => null,
				'parent_slug'        => '',
			)
		);
		add_action( 'admin_menu', array( $this, 'admin_menu' ), 99 );

		add_action('wp_ajax_get_data_from_active_tab_script', array($this, 'get_data_from_active_tab_script'));
		add_action('wp_ajax_nopriv_get_data_from_active_tab_script', array($this, 'get_data_from_active_tab_script'));
	}

	/**
	 * Form action URL
	 */
	private function form_action_url() {
		return apply_filters( 'surecart_client_license_form_action', '' );
	}

	/**
	 * Set the option key.
	 *
	 * If someone wants to override the default generated key.
	 *
	 * @param string $key The option key.
	 */
	public function set_option_key( $key ) {
		$this->option_key = $key;
		return $this;
	}

	/**
	 * Add the admin menu
	 *
	 * @return void
	 */
	public function admin_menu() {
		switch ( $this->menu_args['type'] ) {
			case 'menu':
				$this->create_menu_page();
				break;
			case 'submenu':
				$this->create_submenu_page();
				break;
			case 'options':
				$this->create_options_page();
				break;
		}
	}

	/**
	 * Add license menu page
	 */
	private function create_menu_page() {
		call_user_func(
			'add_menu_page',
			$this->menu_args['page_title'],
			$this->menu_args['menu_title'],
			$this->menu_args['capability'],
			$this->menu_args['menu_slug'],
			array( $this, 'settings_output' ),
			$this->menu_args['icon_url'],
			$this->menu_args['position']
		);
	}

	/**
	 * Add submenu page
	 */
	private function create_submenu_page() {
		call_user_func(
			'add_submenu_page',
			$this->menu_args['parent_slug'],
			$this->menu_args['page_title'],
			$this->menu_args['menu_title'],
			$this->menu_args['capability'],
			$this->menu_args['menu_slug'],
			array( $this, 'settings_output' ),
			$this->menu_args['position']
		);
	}

	/**
	 * Add submenu page
	 */
	private function create_options_page() {
		call_user_func(
			'add_options_page',
			$this->menu_args['page_title'],
			$this->menu_args['menu_title'],
			$this->menu_args['capability'],
			$this->menu_args['menu_slug'],
			array( $this, 'settings_output' ),
			$this->menu_args['position']
		);
	}

	/**
	 * Get all options
	 *
	 * @return array
	 */
	public function get_options() {
		return (array) get_option( $this->option_key, array() );
	}

	/**
	 * Clear out the options.
	 *
	 * @return bool
	 */
	public function clear_options() {
		return update_option( $this->option_key, array() );
	}

	/**
	 * Get a specific option
	 *
	 * @param string $name Option name.
	 *
	 * @return mixed
	 */
	public function get_option( $name ) {
		$options = $this->get_options();
		return isset( $options[ $name ] ) ? $options[ $name ] : null;
	}

	/**
	 * Set the option.
	 *
	 * @param string $name The option name.
	 * @param mixed  $value The option value.
	 *
	 * @return bool
	 */
	public function set_option( $name, $value ) {
		$options          = (array) $this->get_options();
		$options[ $name ] = $value;
		return update_option( $this->option_key, $options );
	}

	/**
	 * The settings page menu output.
	 *
	 * @return void
	 */
	public function settings_output() {
		$this->license_form_submit();

		$this->print_css();

		$activation = $this->get_activation();
		$action     = ! empty( $activation->id ) ? 'deactivate' : 'activate'
		?>

		<?php require_once NEXTBRICKS_PLUGIN_DIR . 'manage-customizations-options.php'; wp_enqueue_media();?>
		<link href="https://fonts.googleapis.com/css2?family=Inter:wght@100..900&display=swap" rel="stylesheet">

		<?php
		$active_tab = get_option('bc_active_tab', 'texts');
		?>

		<div class="bc-body">
			<div class="bc-sidebar-settings-wrapper">
				<input type="hidden" name="bc_active_tab" id="bc_active_tab" value="<?php echo esc_attr(get_option('bc_active_tab', 'texts')); ?>">

				<div class="bc-sidebar">
					<div class="bc-sidebar__top-wrapper">
						<span class="bc-sidebar__caption">Elements</span>

						<span data-tab="texts" class="bc-sidebar__nav-item-wrapper bc-sidebar__nav-top-item-wrapper
						<?php echo $active_tab === 'texts' ? 'bc-sidebar__nav-item-wrapper-active' : ''; ?>">
							<img class="bc-sidebar__nav-item-image" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/nextbricks_texts.png'?>">
							Texts
						</span>

						<span data-tab="buttons" class="bc-sidebar__nav-item-wrapper bc-sidebar__nav-top-item-wrapper
						<?php echo $active_tab === 'buttons' ? 'bc-sidebar__nav-item-wrapper-active' : ''; ?>">
							<img class="bc-sidebar__nav-item-image" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/nextbricks_buttons.png'?>">
							Buttons
						</span>

						<span data-tab="menus" class="bc-sidebar__nav-item-wrapper bc-sidebar__nav-top-item-wrapper
						<?php echo $active_tab === 'menus' ? 'bc-sidebar__nav-item-wrapper-active' : ''; ?>">
							<img class="bc-sidebar__nav-item-image" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/nextbricks_menus.png'?>">
							Menus
						</span>

						<span data-tab="sliders" class="bc-sidebar__nav-item-wrapper bc-sidebar__nav-top-item-wrapper
						<?php echo $active_tab === 'sliders' ? 'bc-sidebar__nav-item-wrapper-active' : ''; ?>">
							<img class="bc-sidebar__nav-item-image" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/nextbricks_sliders.png'?>">
							Sliders
						</span>

						<span data-tab="backgrounds" class="bc-sidebar__nav-item-wrapper bc-sidebar__nav-top-item-wrapper
						<?php echo $active_tab === 'backgrounds' ? 'bc-sidebar__nav-item-wrapper-active' : ''; ?>">
							<img class="bc-sidebar__nav-item-image" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/nextbricks_backgrounds.png'?>">
							Backgrounds
						</span>

						<span data-tab="cursors" class="bc-sidebar__nav-item-wrapper bc-sidebar__nav-top-item-wrapper
						<?php echo $active_tab === 'cursors' ? 'bc-sidebar__nav-item-wrapper-active' : ''; ?>">
							<img class="bc-sidebar__nav-item-image" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/nextbricks_cursors.png'?>">
							Cursors
						</span>

						<span data-tab="medias" class="bc-sidebar__nav-item-wrapper bc-sidebar__nav-top-item-wrapper
						<?php echo $active_tab === 'medias' ? 'bc-sidebar__nav-item-wrapper-active' : ''; ?>">
							<img class="bc-sidebar__nav-item-image" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/nextbricks_medias.png'?>">
							Medias
						</span>

						<span data-tab="cores" class="bc-sidebar__nav-item-wrapper bc-sidebar__nav-top-item-wrapper
						<?php echo $active_tab === 'cores' ? 'bc-sidebar__nav-item-wrapper-active' : ''; ?>">
							<img class="bc-sidebar__nav-item-image" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/nextbricks_cores.png'?>">
							Cores
						</span>

						<div class="bc-sidebar__nav-item-separator"></div>

						<span class="bc-sidebar__caption">Features</span>

						<span data-tab="next-ui" class="bc-sidebar__nav-item-wrapper bc-sidebar__nav-top-item-wrapper
						<?php echo $active_tab === 'next-ui' ? 'bc-sidebar__nav-item-wrapper-active' : ''; ?>">
							<img class="bc-sidebar__nav-item-image" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/nextbricks_nextui.png'?>">
							<span class="has-span">Next UI
							</span>
						</span>

						<span data-tab="animator" class="bc-sidebar__nav-item-wrapper bc-sidebar__nav-top-item-wrapper
						<?php echo $active_tab === 'animator' ? 'bc-sidebar__nav-item-wrapper-active' : ''; ?>">
							<img class="bc-sidebar__nav-item-image" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/nextbricks_animator.png'?>">
							<span class="has-span">Animator
							</span>
						</span>

						<span data-tab="pagetransitions" class="bc-sidebar__nav-item-wrapper bc-sidebar__nav-top-item-wrapper
						<?php echo $active_tab === 'pagetransitions' ? 'bc-sidebar__nav-item-wrapper-active' : ''; ?>">
							<img class="bc-sidebar__nav-item-image" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/nextbricks_pagetransitions.png'?>">
							<span class="has-span">Page Transitions
							</span>
						</span>
					</div>
					
					<div class="bc-sidebar__bottom-wrapper">
						<div class="bc-sidebar__nav-item-separator"></div>

						<span data-tab="assets" class="bc-sidebar__nav-item-wrapper
						<?php echo $active_tab === 'assets' ? 'bc-sidebar__nav-item-wrapper-active' : ''; ?>">
							<img class="bc-sidebar__nav-item-image bc-sidebar__nav-item-image--icon" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/dashboard_sidebar_assets.png'?>">
							Assets
						</span>

						<span data-tab="license" class="bc-sidebar__nav-item-wrapper
						<?php echo $active_tab === 'license' ? 'bc-sidebar__nav-item-wrapper-active' : ''; ?>">
							<img class="bc-sidebar__nav-item-image bc-sidebar__nav-item-image--icon" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/dashboard_sidebar_license.png'?>">
							<span class="bc-sidebar__license-span" data-license-activated="<?php echo get_option('activate_brickscore')?>">
								License
								<span class="bc-sidebar__license-span-inactive">Inactive</span>
								<span class="bc-sidebar__license-span-active">Active</span>
							</span>
						</span>

						<div class="bc-sidebar__nav-item-separator"></div>

						<div class="bc-sidebar__logo-wrapper">
							<img class="bc-sidebar__logo" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/logo-next-bricks.png'?>">
							<a class="bc-sidebar__version" href="https://nextbricks.io/changelog" target="_blank">version <?php echo NEXTBRICKS_VERSION ?></a>
						</div>

						<div class="bc-sidebar__nav-item-separator bc-sidebar__nav-item-separator--mobile"></div>
					</div>
				</div>

				<form class="no-license-form" method="post">
					<div data-tab="texts" class="bc-settings
					<?php echo $active_tab === 'texts' ? 'bc-settings--active' : ''; ?>">
						<div class="bc-settings__wrapper">
							<div class="bc-settings__title-save-wrapper">
								<div class="bc-settings__title-wrapper">
									<img class="bc-settings__icon" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/nextbricks_texts.png'?>">
									<span class="bc-settings__title">Texts</span>
								</div>

								<div class="bc-settings__save">
									<button type="submit" class="panel-bc-submit">Save</button>
								</div>
							</div>

							<div class="wrapper-element-setting-wrapper">
								<input type="hidden" name="bc-dashboard-form"/>

								<div class="search-toggle-all-wrapper">
									<input type="text" id="searchInput" placeholder="Search" autocomplete="off" spellcheck="false">

									<div class="element-setting-wrapper toggle-all-wrapper">
										<span class="heading-element-setting">Toggle All</span>
										<input type="checkbox" id="toggle-all-texts" class="toggle-all" name="panelbctogglealltexts" value="true" <?php checked('true', $bc_toggle_all_texts_enable); ?>/>
										<label for="toggle-all-texts"></label>
									</div>
								</div>

								<div class="element-settings-grid">
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/highlight/" target="_blank" class="heading-element-setting">Highlight</a>
										<input type="checkbox" id="panel-next-highlight" name="panelnexthighlight" value="true" <?php checked('true', $next_highlight_enable); ?>/>
										<label for="panel-next-highlight"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/link-preview/" target="_blank" class="heading-element-setting">Link Preview</a>
										<input type="checkbox" id="panel-next-link-preview" name="panelnextlinkpreview" value="true" <?php checked('true', $next_link_preview_enable); ?>/>
										<label for="panel-next-link-preview"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/read-more/" target="_blank" class="heading-element-setting">Read More</a>
										<input type="checkbox" id="panel-next-read-more" name="panelnextreadmore" value="true" <?php checked('true', $next_read_more_enable); ?>/>
										<label for="panel-next-read-more"></label>
									</div>
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/dynamic-copyright/" target="_blank" class="heading-element-setting">Dynamic Copyright</a>
										<input type="checkbox" id="panel-next-dynamic-copyright" name="panelnextdynamiccopyright" value="true" <?php checked('true', $next_dynamic_copyright_enable); ?>/>
										<label for="panel-next-dynamic-copyright"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/proximity-hover/" target="_blank" class="heading-element-setting">Proximity Hover</a>
										<input type="checkbox" id="panel-next-proximity-hover" name="panelnextproximityhover" value="true" <?php checked('true', $next_proximity_hover_enable); ?>/>
										<label for="panel-next-proximity-hover"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/swap-reading/" target="_blank" class="heading-element-setting">Swap Reading</a>
										<input type="checkbox" id="panel-next-swap-reading" name="panelnextswapreading" value="true" <?php checked('true', $next_swap_reading_enable); ?>/>
										<label for="panel-next-swap-reading"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/variable-font-hover/" target="_blank" class="heading-element-setting">Variable Font Hover</a>
										<input type="checkbox" id="panel-next-variable-font-hover" name="panelnextvariablefonthover" value="true" <?php checked('true', $next_variable_font_hover_enable); ?>/>
										<label for="panel-next-variable-font-hover"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/shading-lines-title/" target="_blank" class="heading-element-setting">Shading Lines Title</a>
										<input type="checkbox" id="panel-next-shading-lines-title" name="panelnextshadinglinestitle" value="true" <?php checked('true', $next_shading_lines_title_enable); ?>/>
										<label for="panel-next-shading-lines-title"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/exchange-title-v2/" target="_blank" class="heading-element-setting">Exchange Title v2</a>
										<input type="checkbox" id="panel-next-exchange-title-v2" name="panelnextexchangetitlev2" value="true" <?php checked('true', $next_exchange_title_v2_enable); ?>/>
										<label for="panel-next-exchange-title-v2"></label>
									</div>
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/text-shimmer/" target="_blank" class="heading-element-setting">Text Shimmer</a>
										<input type="checkbox" id="panel-next-text-shimmer" name="panelnexttextshimmer" value="true" <?php checked('true', $next_text_shimmer_enable); ?>/>
										<label for="panel-next-text-shimmer"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/countdown/" target="_blank" class="heading-element-setting">Countdown</a>
										<input type="checkbox" id="panel-next-countdown" name="panelnextcountdown" value="true" <?php checked('true', $next_countdown_enable); ?>/>
										<label for="panel-next-countdown"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/counter-v2/" target="_blank" class="heading-element-setting">Counter v2</a>
										<input type="checkbox" id="panel-next-counter-v2" name="panelnextcounterv2" value="true" <?php checked('true', $next_counter_v2_enable); ?>/>
										<label for="panel-next-counter-v2"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/blur-reading/" target="_blank" class="heading-element-setting">Blur Reading</a>
										<input type="checkbox" id="panel-bc-blur-reading" name="panelbcblurreading" value="true" <?php checked('true', $bc_blur_reading_enable); ?>/>
										<label for="panel-bc-blur-reading"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/blur-reveal/" target="_blank" class="heading-element-setting">Blur Reveal</a>
										<input type="checkbox" id="panel-bc-blur-reveal" name="panelbcblurreveal" value="true" <?php checked('true', $bc_blur_reveal_enable); ?>/>
										<label for="panel-bc-blur-reveal"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/arc-title/" target="_blank" class="heading-element-setting">Arc Title</a>
										<input type="checkbox" id="panel-bc-circular-title" name="panelbccirculartitle" value="true" <?php checked('true', $bc_circular_title_enable); ?>/>
										<label for="panel-bc-circular-title"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/blade-reveal/" target="_blank" class="heading-element-setting">Blade Reveal</a>
										<input type="checkbox" id="panel-bc-blade-title" name="panelbcbladetitle" value="true" <?php checked('true', $bc_blade_title_enable); ?>/>
										<label for="panel-bc-blade-title"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/blended-hover/" target="_blank" class="heading-element-setting">Blended Hover</a>
										<input type="checkbox" id="panel-bc-blended-title" name="panelbcblendedtitle" value="true" <?php checked('true', $bc_blended_title_enable); ?>/>
										<label for="panel-bc-blended-title"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/counter/" target="_blank" class="heading-element-setting">Counter</a>
										<input type="checkbox" id="panel-bc-counter" name="panelbccounter" value="true" <?php checked('true', $bc_counter_enable); ?>/>
										<label for="panel-bc-counter"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/decode-hover/" target="_blank" class="heading-element-setting">Decode Hover</a>
										<input type="checkbox" id="panel-bc-decode-hover" name="panelbcdecodehover" value="true" <?php checked('true', $bc_decode_hover_enable); ?>/>
										<label for="panel-bc-decode-hover"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/decode-reveal/" target="_blank" class="heading-element-setting">Decode Reveal</a>
										<input type="checkbox" id="panel-bc-decode-reveal" name="panelbcdecodereveal" value="true" <?php checked('true', $bc_decode_reveal_enable); ?>/>
										<label for="panel-bc-decode-reveal"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/exchange-title/" target="_blank" class="heading-element-setting">Exchange Title</a>
										<input type="checkbox" id="panel-bc-exchange-title" name="panelbcexchangetitle" value="true" <?php checked('true', $bc_exchange_title_enable); ?>/>
										<label for="panel-bc-exchange-title"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/fluid-gradient-title/" target="_blank" class="heading-element-setting">Fluid Gradient Title</a>
										<input type="checkbox" id="panel-bc-fluid-gradient" name="panelbcfluidgradient" value="true" <?php checked('true', $bc_fluid_gradient_enable); ?>/>
										<label for="panel-bc-fluid-gradient"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/highlight-hover/" target="_blank" class="heading-element-setting">Highlight Hover</a>
										<input type="checkbox" id="panel-bc-highlight-title" name="panelbchighlighttitle" value="true" <?php checked('true', $bc_highlight_title_enable); ?>/>
										<label for="panel-bc-highlight-title"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/inline-images/" target="_blank" class="heading-element-setting">Inline Images</a>
										<input type="checkbox" id="panel-bc-title-images" name="panelbctitleimages" value="true" <?php checked('true', $bc_title_images_enable); ?>/>
										<label for="panel-bc-title-images"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/letter-launcher/" target="_blank" class="heading-element-setting">Letter Launcher</a>
										<input type="checkbox" id="panel-bc-letter-launcher" name="panelbcletterlauncher" value="true" <?php checked('true', $bc_letter_launcher_enable); ?>/>
										<label for="panel-bc-letter-launcher"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/mask-hover/" target="_blank" class="heading-element-setting">Mask Hover</a>
										<input type="checkbox" id="panel-bc-mask-hover" name="panelbcmaskhover" value="true" <?php checked('true', $bc_mask_hover_enable); ?>/>
										<label for="panel-bc-mask-hover"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/mousefill-title/" target="_blank" class="heading-element-setting">Mousefill Title</a>
										<input type="checkbox" id="panel-bc-mousefill-title" name="panelbcmousefilltitle" value="true" <?php checked('true', $bc_mousefill_title_enable); ?>/>
										<label for="panel-bc-mousefill-title"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/random-letters/" target="_blank" class="heading-element-setting">Random Letters</a>
										<input type="checkbox" id="panel-bc-random-letters" name="panelbcrandomletters" value="true" <?php checked('true', $bc_random_letters_enable); ?>/>
										<label for="panel-bc-random-letters"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/scribble-reveal/" target="_blank" class="heading-element-setting">Scribble Reveal</a>
										<input type="checkbox" id="panel-bc-scribble-title" name="panelbcscribbletitle" value="true" <?php checked('true', $bc_scribble_title_enable); ?>/>
										<label for="panel-bc-scribble-title"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/scroll-cascading/" target="_blank" class="heading-element-setting">Scroll Cascading</a>
										<input type="checkbox" id="panel-bc-scroll-cascading" name="panelbcscrollcascading" value="true" <?php checked('true', $bc_scroll_cascading_enable); ?>/>
										<label for="panel-bc-scroll-cascading"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/scroll-gradient/" target="_blank" class="heading-element-setting">Scroll Gradient</a>
										<input type="checkbox" id="panel-bc-scroll-gradient" name="panelbcscrollgradient" value="true" <?php checked('true', $bc_scroll_gradient_enable); ?>/>
										<label for="panel-bc-scroll-gradient"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/scroll-reading/" target="_blank" class="heading-element-setting">Scroll Reading</a>
										<input type="checkbox" id="panel-bc-scroll-reading" name="panelbcscrollreading" value="true" <?php checked('true', $bc_scroll_reading_enable); ?>/>
										<label for="panel-bc-scroll-reading"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/svg-reveal/" target="_blank" class="heading-element-setting">SVG Reveal</a>
										<input type="checkbox" id="panel-bc-svg-title" name="panelbcsvgtitle" value="true" <?php checked('true', $bc_svg_title_enable); ?>/>
										<label for="panel-bc-svg-title"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/swap-hover/" target="_blank" class="heading-element-setting">Swap Hover</a>
										<input type="checkbox" id="panel-bc-swap-title" name="panelbcswaptitle" value="true" <?php checked('true', $bc_swap_title_enable); ?>/>
										<label for="panel-bc-swap-title"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/3d-swap-hover/" target="_blank" class="heading-element-setting">3D Swap Hover</a>
										<input type="checkbox" id="panel-bc-3dswap-title" name="panelbc3dswaptitle" value="true" <?php checked('true', $bc_3d_swap_title_enable); ?>/>
										<label for="panel-bc-3dswap-title"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/twist-reveal/" target="_blank" class="heading-element-setting">Twist Reveal</a>
										<input type="checkbox" id="panel-bc-twist-title" name="panelbctwisttitle" value="true" <?php checked('true', $bc_twist_title_enable); ?>/>
										<label for="panel-bc-twist-title"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/typed/" target="_blank" class="heading-element-setting">Typed</a>
										<input type="checkbox" id="panel-bc-typed" name="panelbctyped" value="true" <?php checked('true', $bc_typed_enable); ?>/>
										<label for="panel-bc-typed"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/underline-hover/" target="_blank" class="heading-element-setting">Underline Hover</a>
										<input type="checkbox" id="panel-bc-underline-title" name="panelbcunderlinetitle" value="true" <?php checked('true', $bc_underline_title_enable); ?>/>
										<label for="panel-bc-underline-title"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/unfold-hover/" target="_blank" class="heading-element-setting">Unfold Hover</a>
										<input type="checkbox" id="panel-bc-unfold-hover" name="panelbcunfoldhover" value="true" <?php checked('true', $bc_unfold_hover_enable); ?>/>
										<label for="panel-bc-unfold-hover"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/unfold-reveal/" target="_blank" class="heading-element-setting">Unfold Reveal</a>
										<input type="checkbox" id="panel-bc-unfold-reveal" name="panelbcunfoldreveal" value="true" <?php checked('true', $bc_unfold_reveal_enable); ?>/>
										<label for="panel-bc-unfold-reveal"></label>
									</div>
								</div>
							</div>
						</div>
					</div>

					<div data-tab="buttons" class="bc-settings
					<?php echo $active_tab === 'buttons' ? 'bc-settings--active' : ''; ?>">
						<div class="bc-settings__wrapper">
							<div class="bc-settings__title-save-wrapper">
								<div class="bc-settings__title-wrapper">
									<img class="bc-settings__icon" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/nextbricks_buttons.png'?>">
									<span class="bc-settings__title">Buttons</span>
								</div>

								<div class="bc-settings__save">
									<button type="submit" class="panel-bc-submit">Save</button>
								</div>
							</div>

							<div class="wrapper-element-setting-wrapper">
								<input type="hidden" name="bc-dashboard-form"/>

								<div class="search-toggle-all-wrapper">
									<input type="text" id="searchInput" placeholder="Search" autocomplete="off" spellcheck="false">

									<div class="element-setting-wrapper toggle-all-wrapper">
										<span class="heading-element-setting">Toggle All</span>
										<input type="checkbox" id="toggle-all-buttons" class="toggle-all" name="panelbctoggleallbuttons" value="true" <?php checked('true', $bc_toggle_all_buttons_enable); ?>/>
										<label for="toggle-all-buttons"></label>
									</div>
								</div>

								<div class="element-settings-grid">
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/arrow-button-v4/" target="_blank" class="heading-element-setting">Arrow Button v4</a>
										<input type="checkbox" id="panel-next-arrow-button-v4" name="panelnextarrowbuttonv4" value="true" <?php checked('true', $next_arrow_button_v4_enable); ?>/>
										<label for="panel-next-arrow-button-v4"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/arrow-button-v5/" target="_blank" class="heading-element-setting">Arrow Button v5</a>
										<input type="checkbox" id="panel-next-arrow-button-v5" name="panelnextarrowbuttonv5" value="true" <?php checked('true', $next_arrow_button_v5_enable); ?>/>
										<label for="panel-next-arrow-button-v5"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/arrow-button-v6/" target="_blank" class="heading-element-setting">Arrow Button v6</a>
										<input type="checkbox" id="panel-next-arrow-button-v6" name="panelnextarrowbuttonv6" value="true" <?php checked('true', $next_arrow_button_v6_enable); ?>/>
										<label for="panel-next-arrow-button-v6"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/arrow-button-v7/" target="_blank" class="heading-element-setting">Arrow Button v7</a>
										<input type="checkbox" id="panel-next-arrow-button-v7" name="panelnextarrowbuttonv7" value="true" <?php checked('true', $next_arrow_button_v7_enable); ?>/>
										<label for="panel-next-arrow-button-v7"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/arrow-button-v8/" target="_blank" class="heading-element-setting">Arrow Button v8</a>
										<input type="checkbox" id="panel-next-arrow-button-v8" name="panelnextarrowbuttonv8" value="true" <?php checked('true', $next_arrow_button_v8_enable); ?>/>
										<label for="panel-next-arrow-button-v8"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/blurry-button/" target="_blank" class="heading-element-setting">Blurry Button</a>
										<input type="checkbox" id="panel-next-blurry-button" name="panelnextblurrybutton" value="true" <?php checked('true', $next_blurry_button_enable); ?>/>
										<label for="panel-next-blurry-button"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/bubbles-button/" target="_blank" class="heading-element-setting">Bubbles Button</a>
										<input type="checkbox" id="panel-next-bubbles-button" name="panelnextbubblesbutton" value="true" <?php checked('true', $next_bubbles_button_enable); ?>/>
										<label for="panel-next-bubbles-button"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/dot-button/" target="_blank" class="heading-element-setting">Dot Button</a>
										<input type="checkbox" id="panel-next-dot-button" name="panelnextdotbutton" value="true" <?php checked('true', $next_dot_button_enable); ?>/>
										<label for="panel-next-dot-button"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/dot-button-v2/" target="_blank" class="heading-element-setting">Dot Button v2</a>
										<input type="checkbox" id="panel-next-dot-button-v2" name="panelnextdotbuttonv2" value="true" <?php checked('true', $next_dot_button_v2_enable); ?>/>
										<label for="panel-next-dot-button-v2"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/dot-button-v3/" target="_blank" class="heading-element-setting">Dot Button v3</a>
										<input type="checkbox" id="panel-next-dot-button-v3" name="panelnextdotbuttonv3" value="true" <?php checked('true', $next_dot_button_v3_enable); ?>/>
										<label for="panel-next-dot-button-v3"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/flipflop-button/" target="_blank" class="heading-element-setting">Flipflop Button</a>
										<input type="checkbox" id="panel-next-flipflop-button" name="panelnextflipflopbutton" value="true" <?php checked('true', $next_flipflop_button_enable); ?>/>
										<label for="panel-next-flipflop-button"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/flipflop-button-v2/" target="_blank" class="heading-element-setting">Flipflop Button v2</a>
										<input type="checkbox" id="panel-next-flipflop-button-v2" name="panelnextflipflopbuttonv2" value="true" <?php checked('true', $next_flipflop_button_v2_enable); ?>/>
										<label for="panel-next-flipflop-button-v2"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/flipflop-button-v3/" target="_blank" class="heading-element-setting">Flipflop Button v3</a>
										<input type="checkbox" id="panel-next-flipflop-button-v3" name="panelnextflipflopbuttonv3" value="true" <?php checked('true', $next_flipflop_button_v3_enable); ?>/>
										<label for="panel-next-flipflop-button-v3"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/nudge-button/" target="_blank" class="heading-element-setting">Nudge Button</a>
										<input type="checkbox" id="panel-next-nudge-button" name="panelnextnudgebutton" value="true" <?php checked('true', $next_nudge_button_enable); ?>/>
										<label for="panel-next-nudge-button"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/pixels-button/" target="_blank" class="heading-element-setting">Pixels Button</a>
										<input type="checkbox" id="panel-next-pixels-button" name="panelnextpixelsbutton" value="true" <?php checked('true', $next_pixels_button_enable); ?>/>
										<label for="panel-next-pixels-button"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/prism-button-v2/" target="_blank" class="heading-element-setting">Prism Button v2</a>
										<input type="checkbox" id="panel-next-prism-button-v2" name="panelnextprismbuttonv2" value="true" <?php checked('true', $next_prism_button_v2_enable); ?>/>
										<label for="panel-next-prism-button-v2"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/reel-button/" target="_blank" class="heading-element-setting">Reel Button</a>
										<input type="checkbox" id="panel-next-reel-button" name="panelnextreelbutton" value="true" <?php checked('true', $next_reel_button_enable); ?>/>
										<label for="panel-next-reel-button"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/ripple-button-v4/" target="_blank" class="heading-element-setting">Ripple Button v4</a>
										<input type="checkbox" id="panel-next-ripple-button-v4" name="panelnextripplebuttonv4" value="true" <?php checked('true', $next_ripple_button_v4_enable); ?>/>
										<label for="panel-next-ripple-button-v4"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/ripple-button-v5/" target="_blank" class="heading-element-setting">Ripple Button v5</a>
										<input type="checkbox" id="panel-next-ripple-button-v5" name="panelnextripplebuttonv5" value="true" <?php checked('true', $next_ripple_button_v5_enable); ?>/>
										<label for="panel-next-ripple-button-v5"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/ripple-button-v6/" target="_blank" class="heading-element-setting">Ripple Button v6</a>
										<input type="checkbox" id="panel-next-ripple-button-v6" name="panelnextripplebuttonv6" value="true" <?php checked('true', $next_ripple_button_v6_enable); ?>/>
										<label for="panel-next-ripple-button-v6"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/ripple-button-v7/" target="_blank" class="heading-element-setting">Ripple Button v7</a>
										<input type="checkbox" id="panel-next-ripple-button-v7" name="panelnextripplebuttonv7" value="true" <?php checked('true', $next_ripple_button_v7_enable); ?>/>
										<label for="panel-next-ripple-button-v7"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/squeezy-radius-button/" target="_blank" class="heading-element-setting">Squeezy Radius Button</a>
										<input type="checkbox" id="panel-next-squeezy-radius-button" name="panelnextsqueezyradiusbutton" value="true" <?php checked('true', $next_squeezy_radius_button_enable); ?>/>
										<label for="panel-next-squeezy-radius-button"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/stretchy-button/" target="_blank" class="heading-element-setting">Stretchy Button</a>
										<input type="checkbox" id="panel-next-stretchy-button" name="panelnextstretchybutton" value="true" <?php checked('true', $next_stretchy_button_enable); ?>/>
										<label for="panel-next-stretchy-button"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/strip-button/" target="_blank" class="heading-element-setting">Strip Button</a>
										<input type="checkbox" id="panel-next-strip-button" name="panelnextstripbutton" value="true" <?php checked('true', $next_strip_button_enable); ?>/>
										<label for="panel-next-strip-button"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/stripes-button/" target="_blank" class="heading-element-setting">Stripes Button</a>
										<input type="checkbox" id="panel-next-stripes-button" name="panelnextstripesbutton" value="true" <?php checked('true', $next_stripes_button_enable); ?>/>
										<label for="panel-next-stripes-button"></label>
									</div>
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/rainbow-button/" target="_blank" class="heading-element-setting">Rainbow Button</a>
										<input type="checkbox" id="panel-next-rainbow-button" name="panelnextrainbowbutton" value="true" <?php checked('true', $next_rainbow_button_enable); ?>/>
										<label for="panel-next-rainbow-button"></label>
									</div>
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/shiny-button/" target="_blank" class="heading-element-setting">Shiny Button</a>
										<input type="checkbox" id="panel-next-shiny-button" name="panelnextshinybutton" value="true" <?php checked('true', $next_shiny_button_enable); ?>/>
										<label for="panel-next-shiny-button"></label>
									</div>
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/arrow-button-v2/" target="_blank" class="heading-element-setting">Arrow Button v2</a>
										<input type="checkbox" id="panel-next-arrow-button-v2" name="panelnextarrowbuttonv2" value="true" <?php checked('true', $next_arrow_button_v2_enable); ?>/>
										<label for="panel-next-arrow-button-v2"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/arrow-button-v3/" target="_blank" class="heading-element-setting">Arrow Button v3</a>
										<input type="checkbox" id="panel-next-arrow-button-v3" name="panelnextarrowbuttonv3" value="true" <?php checked('true', $next_arrow_button_v3_enable); ?>/>
										<label for="panel-next-arrow-button-v3"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/anyside-button/" target="_blank" class="heading-element-setting">Anyside Button</a>
										<input type="checkbox" id="panel-bc-anyside-button" name="panelbcanysidebutton" value="true" <?php checked('true', $bc_anyside_button_enable); ?>/>
										<label for="panel-bc-anyside-button"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/arrow-button/" target="_blank" class="heading-element-setting">Arrow Button</a>
										<input type="checkbox" id="panel-bc-arrow-button" name="panelbcarrowbutton" value="true" <?php checked('true', $bc_arrow_button_enable); ?>/>
										<label for="panel-bc-arrow-button"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/arrow-icon/" target="_blank" class="heading-element-setting">Arrow Icon</a>
										<input type="checkbox" id="panel-next-arrow-icon" name="panelnextarrowicon" value="true" <?php checked('true', $next_arrow_icon_enable); ?>/>
										<label for="panel-next-arrow-icon"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/crystal-button/" target="_blank" class="heading-element-setting">Crystal Button</a>
										<input type="checkbox" id="panel-bc-crystal-button" name="panelbccrystalbutton" value="true" <?php checked('true', $bc_crystal_button_enable); ?>/>
										<label for="panel-bc-crystal-button"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/glowing-button/" target="_blank" class="heading-element-setting">Glowing Button</a>
										<input type="checkbox" id="panel-bc-glowing-button" name="panelbcglowingbutton" value="true" <?php checked('true', $bc_glowing_button_enable); ?>/>
										<label for="panel-bc-glowing-button"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/layer-button/" target="_blank" class="heading-element-setting">Layer Button</a>
										<input type="checkbox" id="panel-bc-layer-button" name="panelbclayerbutton" value="true" <?php checked('true', $bc_layer_button_enable); ?>/>
										<label for="panel-bc-layer-button"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/marquee-button/" target="_blank" class="heading-element-setting">Marquee Button</a>
										<input type="checkbox" id="panel-bc-marquee-button" name="panelbcmarqueebutton" value="true" <?php checked('true', $bc_marquee_button_enable); ?>/>
										<label for="panel-bc-marquee-button"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/marquee-button-v2/" target="_blank" class="heading-element-setting">Marquee Button v2</a>
										<input type="checkbox" id="panel-next-marquee-button-v2" name="panelnextmarqueebuttonv2" value="true" <?php checked('true', $next_marquee_button_v2_enable); ?>/>
										<label for="panel-next-marquee-button-v2"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/mask-button/" target="_blank" class="heading-element-setting">Mask Button</a>
										<input type="checkbox" id="panel-bc-mask-button" name="panelbcmaskbutton" value="true" <?php checked('true', $bc_mask_button_enable); ?>/>
										<label for="panel-bc-mask-button"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/microbox-button/" target="_blank" class="heading-element-setting">Microbox Button</a>
										<input type="checkbox" id="panel-bc-microbox-button" name="panelbcmicroboxbutton" value="true" <?php checked('true', $bc_microbox_button_enable); ?>/>
										<label for="panel-bc-microbox-button"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/neumorphic-button/" target="_blank" class="heading-element-setting">Neumorphic Button</a>
										<input type="checkbox" id="panel-bc-neumorphic-button" name="panelbcneumorphicbutton" value="true" <?php checked('true', $bc_neumorphic_button_enable); ?>/>
										<label for="panel-bc-neumorphic-button"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/prism-button/" target="_blank" class="heading-element-setting">Prism Button</a>
										<input type="checkbox" id="panel-bc-prism-button" name="panelbcprismbutton" value="true" <?php checked('true', $bc_prism_button_enable); ?>/>
										<label for="panel-bc-prism-button"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/ripple-button/" target="_blank" class="heading-element-setting">Ripple Button</a>
										<input type="checkbox" id="panel-bc-ripple-button" name="panelbcripplebutton" value="true" <?php checked('true', $bc_ripple_button_enable); ?>/>
										<label for="panel-bc-ripple-button"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/ripple-button-v2/" target="_blank" class="heading-element-setting">Ripple Button v2</a>
										<input type="checkbox" id="panel-next-ripple-button-v2" name="panelnextripplebuttonv2" value="true" <?php checked('true', $next_ripple_button_v2_enable); ?>/>
										<label for="panel-next-ripple-button-v2"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/ripple-button-v3/" target="_blank" class="heading-element-setting">Ripple Button v3</a>
										<input type="checkbox" id="panel-next-ripple-button-v3" name="panelnextripplebuttonv3" value="true" <?php checked('true', $next_ripple_button_v3_enable); ?>/>
										<label for="panel-next-ripple-button-v3"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/split-button/" target="_blank" class="heading-element-setting">Split Button</a>
										<input type="checkbox" id="panel-bc-split-button" name="panelbcsplitbutton" value="true" <?php checked('true', $bc_split_button_enable); ?>/>
										<label for="panel-bc-split-button"></label>
									</div>
								</div>
							</div>
						</div>
					</div>

					<div data-tab="menus" class="bc-settings
					<?php echo $active_tab === 'menus' ? 'bc-settings--active' : ''; ?>">
						<div class="bc-settings__wrapper">
							<div class="bc-settings__title-save-wrapper">
								<div class="bc-settings__title-wrapper">
									<img class="bc-settings__icon" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/nextbricks_menus.png'?>">
									<span class="bc-settings__title">Menus</span>
								</div>

								<div class="bc-settings__save">
									<button type="submit" class="panel-bc-submit">Save</button>
								</div>
							</div>

							<div class="wrapper-element-setting-wrapper">
								<input type="hidden" name="bc-dashboard-form"/>

								<div class="search-toggle-all-wrapper">
									<input type="text" id="searchInput" placeholder="Search" autocomplete="off" spellcheck="false">

									<div class="element-setting-wrapper toggle-all-wrapper">
										<span class="heading-element-setting">Toggle All</span>
										<input type="checkbox" id="toggle-all-menus" class="toggle-all" name="panelbctoggleallmenus" value="true" <?php checked('true', $bc_toggle_all_menus_enable); ?>/>
										<label for="toggle-all-menus"></label>
									</div>
								</div>

								<div class="element-settings-grid">
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/dynamic-island" target="_blank" class="heading-element-setting">Dynamic Island</a>
										<input type="checkbox" id="panel-next-dynamic-island" name="panelnextdynamicisland" value="true" <?php checked('true', $next_dynamic_island_enable); ?>/>
										<label for="panel-next-dynamic-island"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/gooey-nav" target="_blank" class="heading-element-setting">Gooey Nav</a>
										<input type="checkbox" id="panel-next-gooey-nav" name="panelnextgooeynav" value="true" <?php checked('true', $next_gooey_nav_enable); ?>/>
										<label for="panel-next-gooey-nav"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/focus-nav" target="_blank" class="heading-element-setting">Focus Nav</a>
										<input type="checkbox" id="panel-next-focus-nav" name="panelnextfocusnav" value="true" <?php checked('true', $next_focus_nav_enable); ?>/>
										<label for="panel-next-focus-nav"></label>
									</div>
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/dock" target="_blank" class="heading-element-setting">Dock</a>
										<input type="checkbox" id="panel-nb-dock-menu" name="panelnextdockmenu" value="true" <?php checked('true', $next_dock_menu_enable); ?>/>
										<label for="panel-nb-dock-menu"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/cta-nav" target="_blank" class="heading-element-setting">CTA Nav</a>
										<input type="checkbox" id="panel-nb-cta-nav" name="panelnextctanav" value="true" <?php checked('true', $next_cta_nav_enable); ?>/>
										<label for="panel-nb-cta-nav"></label>
									</div>
									
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/circular-menu" target="_blank" class="heading-element-setting">Circular Menu</a>
										<input type="checkbox" id="panel-nb-circular-menu" name="panelnextcircularmenu" value="true" <?php checked('true', $next_circular_menu_enable); ?>/>
										<label for="panel-nb-circular-menu"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/burger" target="_blank" class="heading-element-setting">Burger</a>
										<input type="checkbox" id="panel-bc-coreburguer" name="panelbccoreburguer" value="true" <?php checked('true', $bc_core_burguer_enable); ?>/>
										<label for="panel-bc-coreburguer"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/drawer/" target="_blank" class="heading-element-setting">Drawer</a>
										<input type="checkbox" id="panel-bc-drawer" name="panelbcdrawer" value="true" <?php checked('true', $bc_drawer_enable); ?>/>
										<label for="panel-bc-drawer"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/dropdown/" target="_blank" class="heading-element-setting">Dropdown</a>
										<input type="checkbox" id="panel-next-dropdown" name="panelnextdropdown" value="true" <?php checked('true', $next_dropdown_enable); ?>/>
										<label for="panel-next-dropdown"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/dropdown-megamenu/" target="_blank" class="heading-element-setting">Dropdown megaMenu</a>
										<input type="checkbox" id="panel-next-dropdown-mega-menu" name="panelnextdropdownmegamenu" value="true" <?php checked('true', $next_dropdown_mega_menu_enable); ?>/>
										<label for="panel-next-dropdown-mega-menu"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/expanding-menu/" target="_blank" class="heading-element-setting">Expanding Menu</a>
										<input type="checkbox" id="panel-bc-expandingmenu" name="panelbcexpandingmenu" value="true" <?php checked('true', $bc_expanding_menu_enable); ?>/>
										<label for="panel-bc-expandingmenu"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/morphing-nav/" target="_blank" class="heading-element-setting">Morphing Nav</a>
										<input type="checkbox" id="panel-bc-morphingmenu" name="panelbcmorphingmenu" value="true" <?php checked('true', $bc_morphing_menu_enable); ?>/>
										<label for="panel-bc-morphingmenu"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/multi-offcanvas-menu/" target="_blank" class="heading-element-setting">Multi OffCanvas</a>
										<input type="checkbox" id="panel-bc-multioffcanvas" name="panelbcmultioffcanvas" value="true" <?php checked('true', $bc_multioffcanvas_enable); ?>/>
										<label for="panel-bc-multioffcanvas"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/offcanvas-menu/" target="_blank" class="heading-element-setting">OffCanvas Menu</a>
										<input type="checkbox" id="panel-bc-offcanvasmenu" name="panelbcoffcanvasmenu" value="true" <?php checked('true', $bc_offcanvas_menu_enable); ?>/>
										<label for="panel-bc-offcanvasmenu"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/overlay-menu/" target="_blank" class="heading-element-setting">Overlay Menu</a>
										<input type="checkbox" id="panel-bc-simplemenu" name="panelbcsimplemenu" value="true" <?php checked('true', $bc_simple_menu_enable); ?>/>
										<label for="panel-bc-simplemenu"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/sticky-header/" target="_blank" class="heading-element-setting">Sticky Header</a>
										<input type="checkbox" id="panel-bc-stickynavmenu" name="panelbcstickynavmenu" value="true" <?php checked('true', $bc_stickynav_menu_enable); ?>/>
										<label for="panel-bc-stickynavmenu"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/stripe-megamenu/" target="_blank" class="heading-element-setting">Stripe megaMenu</a>
										<input type="checkbox" id="panel-bc-stripemegamenu" name="panelbcstripemegamenu" value="true" <?php checked('true', $bc_stripe_megamenu_enable); ?>/>
										<label for="panel-bc-stripemegamenu"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/wrap-menu/" target="_blank" class="heading-element-setting">Wrap Menu</a>
										<input type="checkbox" id="panel-bc-overlaymenu" name="panelbcoverlaymenu" value="true" <?php checked('true', $bc_overlay_menu_enable); ?>/>
										<label for="panel-bc-overlaymenu"></label>
									</div>
								</div>
							</div>
						</div>
					</div>	

					<div data-tab="sliders" class="bc-settings
					<?php echo $active_tab === 'sliders' ? 'bc-settings--active' : ''; ?>">
						<div class="bc-settings__wrapper">
							<div class="bc-settings__title-save-wrapper">
								<div class="bc-settings__title-wrapper">
									<img class="bc-settings__icon" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/nextbricks_sliders.png'?>">
									<span class="bc-settings__title">Sliders</span>
								</div>

								<div class="bc-settings__save">
									<button type="submit" class="panel-bc-submit">Save</button>
								</div>
							</div>

							<div class="wrapper-element-setting-wrapper">
								<input type="hidden" name="bc-dashboard-form"/>

								<div class="search-toggle-all-wrapper">
									<input type="text" id="searchInput" placeholder="Search" autocomplete="off" spellcheck="false">

									<div class="element-setting-wrapper toggle-all-wrapper">
										<span class="heading-element-setting">Toggle All</span>
										<input type="checkbox" id="toggle-all-sliders" class="toggle-all" name="panelbctoggleallsliders" value="true" <?php checked('true', $bc_toggle_all_sliders_enable); ?>/>
										<label for="toggle-all-sliders"></label>
									</div>
								</div>

								<div class="element-settings-grid">
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/stories/" target="_blank" class="heading-element-setting">Stories</a>
										<input type="checkbox" id="panel-next-stories" name="panelnextstories" value="true" <?php checked('true', $next_stories_enable); ?>/>
										<label for="panel-next-stories"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/thumbnail-slider/" target="_blank" class="heading-element-setting">Thumbnail Slider</a>
										<input type="checkbox" id="panel-next-thumbnail-slider" name="panelnextthumbnailslider" value="true" <?php checked('true', $next_thumbnail_slider_enable); ?>/>
										<label for="panel-next-thumbnail-slider"></label>
									</div>
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/infinite-slider/" target="_blank" class="heading-element-setting">Infinite Slider</a>
										<input type="checkbox" id="panel-next-infinite-slider" name="panelnextinfiniteslider" value="true" <?php checked('true', $next_infinite_slider_enable); ?>/>
										<label for="panel-next-infinite-slider"></label>
									</div>
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/parallax-slider/" target="_blank" class="heading-element-setting">Parallax Slider</a>
										<input type="checkbox" id="panel-next-parallax-slider" name="panelnextparallaxslider" value="true" <?php checked('true', $next_parallax_slider_enable); ?>/>
										<label for="panel-next-parallax-slider"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/next-slider/" target="_blank" class="heading-element-setting">Next Slider</a>
										<input type="checkbox" id="panel-bc-core-slider" name="panelbccoreslider" value="true" <?php checked('true', $bc_core_slider_enable); ?>/>
										<label for="panel-bc-core-slider"></label>
									</div>
								</div>
							</div>
						</div>
					</div>	

					<div data-tab="backgrounds" class="bc-settings
					<?php echo $active_tab === 'backgrounds' ? 'bc-settings--active' : ''; ?>">
						<div class="bc-settings__wrapper">
							<div class="bc-settings__title-save-wrapper">
								<div class="bc-settings__title-wrapper">
									<img class="bc-settings__icon" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/nextbricks_backgrounds.png'?>">
									<span class="bc-settings__title">Backgrounds</span>
								</div>

								<div class="bc-settings__save">
									<button type="submit" class="panel-bc-submit">Save</button>
								</div>
							</div>

							<div class="wrapper-element-setting-wrapper">
								<input type="hidden" name="bc-dashboard-form"/>

								<div class="search-toggle-all-wrapper">
									<input type="text" id="searchInput" placeholder="Search" autocomplete="off" spellcheck="false">

									<div class="element-setting-wrapper toggle-all-wrapper">
										<span class="heading-element-setting">Toggle All</span>
										<input type="checkbox" id="toggle-all-backgrounds" class="toggle-all" name="panelbctoggleallbackgrounds" value="true" <?php checked('true', $bc_toggle_all_backgrounds_enable); ?>/>
										<label for="toggle-all-backgrounds"></label>
									</div>
								</div>

								<div class="element-settings-grid">									
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/figures/" target="_blank" class="heading-element-setting">Figures</a>
										<input type="checkbox" id="panel-next-figures" name="panelnextfigures" value="true" <?php checked('true', $next_figures_enable); ?>/>
										<label for="panel-next-figures"></label>
									</div>
									
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/interactive-lines-v2/" target="_blank" class="heading-element-setting">Interactive Lines v2</a>
										<input type="checkbox" id="panel-next-interactive-lines-v2" name="panelnextinteractivelinesv2" value="true" <?php checked('true', $next_interactive_lines_v2_enable); ?>/>
										<label for="panel-next-interactive-lines-v2"></label>
									</div>
									
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/interactive-lines-v3/" target="_blank" class="heading-element-setting">Interactive Lines v3</a>
										<input type="checkbox" id="panel-next-interactive-lines-v3" name="panelnextinteractivelinesv3" value="true" <?php checked('true', $next_interactive_lines_v3_enable); ?>/>
										<label for="panel-next-interactive-lines-v3"></label>
									</div>
									
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/interactive-lines-v4/" target="_blank" class="heading-element-setting">Interactive Lines v4</a>
										<input type="checkbox" id="panel-next-interactive-lines-v4" name="panelnextinteractivelinesv4" value="true" <?php checked('true', $next_interactive_lines_v4_enable); ?>/>
										<label for="panel-next-interactive-lines-v4"></label>
									</div>
									
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/iridescence/" target="_blank" class="heading-element-setting">Iridescence</a>
										<input type="checkbox" id="panel-next-iridescence" name="panelnextiridescence" value="true" <?php checked('true', $next_iridescence_enable); ?>/>
										<label for="panel-next-iridescence"></label>
									</div>
									
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/overlay-shadows/" target="_blank" class="heading-element-setting">Overlay Shadows</a>
										<input type="checkbox" id="panel-next-overlay-shadows" name="panelnextoverlayshadows" value="true" <?php checked('true', $next_overlay_shadows_enable); ?>/>
										<label for="panel-next-overlay-shadows"></label>
									</div>
									
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/scrolling-background/" target="_blank" class="heading-element-setting">Scrolling Background</a>
										<input type="checkbox" id="panel-next-scrolling-background" name="panelnextscrollingbackground" value="true" <?php checked('true', $next_scrolling_background_enable); ?>/>
										<label for="panel-next-scrolling-background"></label>
									</div>
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/tiles-v2/" target="_blank" class="heading-element-setting">Tiles v2</a>
										<input type="checkbox" id="panel-next-tiles-v2" name="panelnexttilesv2" value="true" <?php checked('true', $next_tiles_v2_enable); ?>/>
										<label for="panel-next-tiles-v2"></label>
									</div>
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/interactive-lines/" target="_blank" class="heading-element-setting">Interactive Lines</a>
										<input type="checkbox" id="panel-next-interactive-lines" name="panelnextinteractivelines" value="true" <?php checked('true', $next_interactive_lines_enable); ?>/>
										<label for="panel-next-interactive-lines"></label>
									</div>
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/fluids/" target="_blank" class="heading-element-setting">Fluids</a>
										<input type="checkbox" id="panel-nb-fluids" name="panelnextfluids" value="true" <?php checked('true', $next_fluids_enable); ?>/>
										<label for="panel-nb-fluids"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/meteors/" target="_blank" class="heading-element-setting">Meteors</a>
										<input type="checkbox" id="panel-nb-meteors" name="panelnextmeteors" value="true" <?php checked('true', $next_meteors_enable); ?>/>
										<label for="panel-nb-meteors"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/tiles/" target="_blank" class="heading-element-setting">Tiles</a>
										<input type="checkbox" id="panel-nb-tiles" name="panelnexttiles" value="true" <?php checked('true', $next_tiles_enable); ?>/>
										<label for="panel-nb-tiles"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/border-beam/" target="_blank" class="heading-element-setting">Border Beam</a>
										<input type="checkbox" id="panel-nb-border-beam" name="panelnextborderbeam" value="true" <?php checked('true', $next_border_beam_enable); ?>/>
										<label for="panel-nb-border-beam"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/bubbles/" target="_blank" class="heading-element-setting">Bubbles</a>
										<input type="checkbox" id="panel-bc-bubbles" name="panelbcbubbles" value="true" <?php checked('true', $bc_bubbles_enable); ?>/>
										<label for="panel-bc-bubbles"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/generatorism/" target="_blank" class="heading-element-setting">Generatorism</a>
										<input type="checkbox" id="panel-bc-generatorism" name="panelbcgeneratorism" value="true" <?php checked('true', $bc_generatorism_enable); ?>/>
										<label for="panel-bc-generatorism"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/gradiently/" target="_blank" class="heading-element-setting">Gradiently</a>
										<input type="checkbox" id="panel-bc-gradiently" name="panelbcgradiently" value="true" <?php checked('true', $bc_gradiently_enable); ?>/>
										<label for="panel-bc-gradiently"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/grainy/" target="_blank" class="heading-element-setting">Grainy</a>
										<input type="checkbox" id="panel-bc-grained" name="panelbcgrained" value="true" <?php checked('true', $bc_grained_enable); ?>/>
										<label for="panel-bc-grained"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/particles/" target="_blank" class="heading-element-setting">Particles</a>
										<input type="checkbox" id="panel-next-particles" name="panelnextparticles" value="true" <?php checked('true', $next_particles_enable); ?>/>
										<label for="panel-next-particles"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/physics/" target="_blank" class="heading-element-setting">Physics</a>
										<input type="checkbox" id="panel-next-physics" name="panelnextphysics" value="true" <?php checked('true', $next_physics_enable); ?>/>
										<label for="panel-next-physics"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/spotlight/" target="_blank" class="heading-element-setting">Spotlight</a>
										<input type="checkbox" id="panel-bc-spotlight" name="panelbcspotlight" value="true" <?php checked('true', $bc_spotlight_enable); ?>/>
										<label for="panel-bc-spotlight"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/spotlight-v2/" class="heading-element-setting">Spotlight v2</a>
										<input type="checkbox" id="panel-next-spotlight-v2" name="panelnextspotlightv2" value="true" <?php checked('true', $next_spotlight_v2_enable); ?>/>
										<label for="panel-next-spotlight-v2"></label>
									</div>
								</div>
							</div>
						</div>
					</div>	

					<div data-tab="cursors" class="bc-settings
					<?php echo $active_tab === 'cursors' ? 'bc-settings--active' : ''; ?>">
						<div class="bc-settings__wrapper">
							<div class="bc-settings__title-save-wrapper">
								<div class="bc-settings__title-wrapper">
									<img class="bc-settings__icon" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/nextbricks_sliders.png'?>">
									<span class="bc-settings__title">Cursors</span>
								</div>

								<div class="bc-settings__save">
									<button type="submit" class="panel-bc-submit">Save</button>
								</div>
							</div>

							<div class="wrapper-element-setting-wrapper">
								<input type="hidden" name="bc-dashboard-form"/>

								<div class="search-toggle-all-wrapper">
									<input type="text" id="searchInput" placeholder="Search" autocomplete="off" spellcheck="false">

									<div class="element-setting-wrapper toggle-all-wrapper">
										<span class="heading-element-setting">Toggle All</span>
										<input type="checkbox" id="toggle-all-cursors" class="toggle-all" name="panelbctoggleallcursors" value="true" <?php checked('true', $bc_toggle_all_cursors_enable); ?>/>
										<label for="toggle-all-cursors"></label>
									</div>
								</div>

								<div class="element-settings-grid">
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/crosshair/" target="_blank" class="heading-element-setting">Crosshair</a>
										<input type="checkbox" id="panel-next-crosshair" name="panelnextcrosshair" value="true" <?php checked('true', $next_crosshair_enable); ?>/>
										<label for="panel-next-crosshair"></label>
									</div>
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/cursor-trail/" target="_blank" class="heading-element-setting">Cursor Trail</a>
										<input type="checkbox" id="panel-next-cursor-trail" name="panelnextcursortrail" value="true" <?php checked('true', $next_cursor_trail_enable); ?>/>
										<label for="panel-next-cursor-trail"></label>
									</div>
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/mouse-parallax/" target="_blank" class="heading-element-setting">Mouse Parallax</a>
										<input type="checkbox" id="panel-next-mouse-parallax" name="panelnextmouseparallax" value="true" <?php checked('true', $next_mouse_parallax_enable); ?>/>
										<label for="panel-next-mouse-parallax"></label>
									</div>
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/cursor-slide/" target="_blank" class="heading-element-setting">Cursor Slide</a>
										<input type="checkbox" id="panel-nb-cursor-slide" name="panelnextcursorslide" value="true" <?php checked('true', $next_cursor_slide_enable); ?>/>
										<label for="panel-nb-cursor-slide"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/designer-cursor/" target="_blank" class="heading-element-setting">Designer Cursor</a>
										<input type="checkbox" id="panel-nb-designer-cursor" name="panelnextdesignercursor" value="true" <?php checked('true', $next_designer_cursor_enable); ?>/>
										<label for="panel-nb-designer-cursor"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/next-cursor/" target="_blank" class="heading-element-setting">Next Cursor</a>
										<input type="checkbox" id="panel-bc-cursor" name="panelbccursor" value="true" <?php checked('true', $bc_cursor_enable); ?>/>
										<label for="panel-bc-cursor"></label>
									</div>
								</div>
							</div>
						</div>
					</div>	

					<div data-tab="medias" class="bc-settings
					<?php echo $active_tab === 'medias' ? 'bc-settings--active' : ''; ?>">
						<div class="bc-settings__wrapper">
							<div class="bc-settings__title-save-wrapper">
								<div class="bc-settings__title-wrapper">
									<img class="bc-settings__icon" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/nextbricks_medias.png'?>">
									<span class="bc-settings__title">Medias</span>
								</div>

								<div class="bc-settings__save">
									<button type="submit" class="panel-bc-submit">Save</button>
								</div>
							</div>

							<div class="wrapper-element-setting-wrapper">
								<input type="hidden" name="bc-dashboard-form"/>

								<div class="search-toggle-all-wrapper">
									<input type="text" id="searchInput" placeholder="Search" autocomplete="off" spellcheck="false">

									<div class="element-setting-wrapper toggle-all-wrapper">
										<span class="heading-element-setting">Toggle All</span>
										<input type="checkbox" id="toggle-all-medias" class="toggle-all" name="panelbctoggleallmedias" value="true" <?php checked('true', $bc_toggle_all_medias_enable); ?>/>
										<label for="toggle-all-medias"></label>
									</div>
								</div>

								<div class="element-settings-grid">
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/before-after-image-v2/" target="_blank" class="heading-element-setting">Before/After v2</a>
										<input type="checkbox" id="panel-next-beforeafter-image-v2" name="panelnextbeforeafterimagev2" value="true" <?php checked('true', $next_beforeafter_image_v2_enable); ?>/>
										<label for="panel-next-beforeafter-image-v2"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/image-reveal-v2/" target="_blank" class="heading-element-setting">Image Reveal v2</a>
										<input type="checkbox" id="panel-next-image-reveal-v2" name="panelnextimagerevealv2" value="true" <?php checked('true', $next_image_reveal_v2_enable); ?>/>
										<label for="panel-next-image-reveal-v2"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/image-reveal-v3/" target="_blank" class="heading-element-setting">Image Reveal v3</a>
										<input type="checkbox" id="panel-next-image-reveal-v3" name="panelnextimagerevealv3" value="true" <?php checked('true', $next_image_reveal_v3_enable); ?>/>
										<label for="panel-next-image-reveal-v3"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/image-reveal-v4/" target="_blank" class="heading-element-setting">Image Reveal v4</a>
										<input type="checkbox" id="panel-next-image-reveal-v4" name="panelnextimagerevealv4" value="true" <?php checked('true', $next_image_reveal_v4_enable); ?>/>
										<label for="panel-next-image-reveal-v4"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/lightbox/" target="_blank" class="heading-element-setting">Lightbox</a>
										<input type="checkbox" id="panel-next-lightbox" name="panelnextlightbox" value="true" <?php checked('true', $next_lightbox_enable); ?>/>
										<label for="panel-next-lightbox"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/scrolling-gallery/" target="_blank" class="heading-element-setting">Scrolling Gallery</a>
										<input type="checkbox" id="panel-next-scrolling-gallery" name="panelnextscrollinggallery" value="true" <?php checked('true', $next_scrolling_gallery_enable); ?>/>
										<label for="panel-next-scrolling-gallery"></label>
									</div>
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/image-accordion/" target="_blank" class="heading-element-setting">Image Accordion</a>
										<input type="checkbox" id="panel-next-image-accordion" name="panelnextimageaccordion" value="true" <?php checked('true', $next_image_accordion_enable); ?>/>
										<label for="panel-next-image-accordion"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/before-after-image/" target="_blank" class="heading-element-setting">Before/After</a>
										<input type="checkbox" id="panel-bc-beforeafterimage" name="panelbcbeforeafterimage" value="true" <?php checked('true', $bc_beforeafter_image_enable); ?>/>
										<label for="panel-bc-beforeafterimage"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/image-hotspots/" target="_blank" class="heading-element-setting">Image Hotspots</a>
										<input type="checkbox" id="panel-bc-imagehotspots" name="panelbcimagehotspots" value="true" <?php checked('true', $bc_image_hotspots_enable); ?>/>
										<label for="panel-bc-imagehotspots"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/image-reveal/" target="_blank" class="heading-element-setting">Image Reveal</a>
										<input type="checkbox" id="panel-bc-image-reveal" name="panelbcimagereveal" value="true" <?php checked('true', $bc_image_reveal_enable); ?>/>
										<label for="panel-bc-image-reveal"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/media/" target="_blank" class="heading-element-setting">Media</a>
										<input type="checkbox" id="panel-bc-media" name="panelbcmedia" value="true" <?php checked('true', $bc_media_enable); ?>/>
										<label for="panel-bc-media"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/parallax/" target="_blank" class="heading-element-setting">Parallax</a>
										<input type="checkbox" id="panel-bc-parallax" name="panelbcparallax" value="true" <?php checked('true', $bc_parallax_enable); ?>/>
										<label for="panel-bc-parallax"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/video-sequence/" target="_blank" class="heading-element-setting">Video Sequence</a>
										<input type="checkbox" id="panel-bc-video-sequence" name="panelbcvideosequence" value="true" <?php checked('true', $bc_video_sequence_enable); ?>/>
										<label for="panel-bc-video-sequence"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/zoom-lens" target="_blank" class="heading-element-setting">Zoom Lens</a>
										<input type="checkbox" id="panel-bc-zoom-lens" name="panelbczoomlens" value="true" <?php checked('true', $bc_zoom_lens_enable); ?>/>
										<label for="panel-bc-zoom-lens"></label>
									</div>
								</div>
							</div>
						</div>
					</div>	

					<div data-tab="cores" class="bc-settings
					<?php echo $active_tab === 'cores' ? 'bc-settings--active' : ''; ?>">
						<div class="bc-settings__wrapper">
							<div class="bc-settings__title-save-wrapper">
								<div class="bc-settings__title-wrapper">
									<img class="bc-settings__icon" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/nextbricks_cores.png'?>">
									<span class="bc-settings__title">Cores</span>
								</div>

								<div class="bc-settings__save">
									<button type="submit" class="panel-bc-submit">Save</button>
								</div>
							</div>

							<div class="wrapper-element-setting-wrapper">
								<input type="hidden" name="bc-dashboard-form"/>

								<div class="search-toggle-all-wrapper">
									<input type="text" id="searchInput" placeholder="Search" autocomplete="off" spellcheck="false">

									<div class="element-setting-wrapper toggle-all-wrapper">
										<span class="heading-element-setting">Toggle All</span>
										<input type="checkbox" id="toggle-all-cores" class="toggle-all" name="panelbctoggleallcores" value="true" <?php checked('true', $bc_toggle_all_cores_enable); ?>/>
										<label for="toggle-all-cores"></label>
									</div>
								</div>

								<div class="element-settings-grid">
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/audio-player/" target="_blank" class="heading-element-setting">Audio Player</a>
										<input type="checkbox" id="panel-next-audio-player" name="panelnextaudioplayer" value="true" <?php checked('true', $next_audio_player_enable); ?>/>
										<label for="panel-next-audio-player"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/looping-tabs/" target="_blank" class="heading-element-setting">Looping Tabs</a>
										<input type="checkbox" id="panel-next-looping-tabs" name="panelnextloopingtabs" value="true" <?php checked('true', $next_looping_tabs_enable); ?>/>
										<label for="panel-next-looping-tabs"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/site-loader/" target="_blank" class="heading-element-setting">Site Loader</a>
										<input type="checkbox" id="panel-next-site-loader" name="panelnextsiteloader" value="true" <?php checked('true', $next_site_loader_enable); ?>/>
										<label for="panel-next-site-loader"></label>
									</div>
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/progress-bar-v2/" target="_blank" class="heading-element-setting">Progress Bar v2</a>
										<input type="checkbox" id="panel-next-progress-bar-v2" name="panelnextprogressbarv2" value="true" <?php checked('true', $next_progress_bar_v2_enable); ?>/>
										<label for="panel-next-progress-bar-v2"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/decode-card/" target="_blank" class="heading-element-setting">Decode Card</a>
										<input type="checkbox" id="panel-next-decode-card" name="panelnextdecodecard" value="true" <?php checked('true', $next_decode_card_enable); ?>/>
										<label for="panel-next-decode-card"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/distorsion-tabs/" target="_blank" class="heading-element-setting">Distorsion Tabs</a>
										<input type="checkbox" id="panel-next-distorsion-tabs" name="panelnextdistortionstabs" value="true" <?php checked('true', $next_distorsion_tabs_enable); ?>/>
										<label for="panel-next-distorsion-tabs"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/toolteam/" target="_blank" class="heading-element-setting">Toolteam</a>
										<input type="checkbox" id="panel-next-toolteam" name="panelnexttoolteam" value="true" <?php checked('true', $next_toolteam_enable); ?>/>
										<label for="panel-next-toolteam"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/toast/" target="_blank" class="heading-element-setting">Toast</a>
										<input type="checkbox" id="panel-next-toast" name="panelnexttoast" value="true" <?php checked('true', $next_toast_enable); ?>/>
										<label for="panel-next-toast"></label>
									</div>
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/pixels-shimmer-card/" target="_blank" class="heading-element-setting">Pixels Shimmer Card</a>
										<input type="checkbox" id="panel-next-pixels-shimmer-card" name="panelnextpixelsshimmercard" value="true" <?php checked('true', $next_pixels_shimmer_card_enable); ?>/>
										<label for="panel-next-pixels-shimmer-card"></label>
									</div>
									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/interactive-divider/" target="_blank" class="heading-element-setting">Interactive Divider</a>
										<input type="checkbox" id="panel-nb-interactive-divider" name="panelnextinteractivedivider" value="true" <?php checked('true', $next_interactive_divider_enable); ?>/>
										<label for="panel-nb-interactive-divider"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/progress-bar/" target="_blank" class="heading-element-setting">Progress Bar</a>
										<input type="checkbox" id="panel-nb-progress-bar" name="panelnextprogressbar" value="true" <?php checked('true', $next_progress_bar_enable); ?>/>
										<label for="panel-nb-progress-bar"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/cookies/" target="_blank" class="heading-element-setting">Cookies</a>
										<input type="checkbox" id="panel-nb-cookies" name="panelnextcookies" value="true" <?php checked('true', $next_cookies_enable); ?>/>
										<label for="panel-nb-cookies"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/qr-code/" target="_blank" class="heading-element-setting">QR Code</a>
										<input type="checkbox" id="panel-nb-qr-code" name="panelnextqrcode" value="true" <?php checked('true', $next_qr_code_enable); ?>/>
										<label for="panel-nb-qr-code"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/stacking-cards/" target="_blank" class="heading-element-setting">Stacking Cards</a>
										<input type="checkbox" id="panel-nb-stacking-cards" name="panelnextstackingcards" value="true" <?php checked('true', $next_stacking_cards_enable); ?>/>
										<label for="panel-nb-stacking-cards"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/inverted-corners/" target="_blank" class="heading-element-setting">Inverted Corner</a>
										<input type="checkbox" id="panel-nb-inverted-corners" name="panelnextinvertedcorners" value="true" <?php checked('true', $next_inverted_corners_enable); ?>/>
										<label for="panel-nb-inverted-corners"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/back-to-top/" target="_blank" class="heading-element-setting">Back to Top</a>
										<input type="checkbox" id="panel-bc-back-to-top" name="panelbcbacktotop" value="true" <?php checked('true', $bc_back_to_top_enable); ?>/>
										<label for="panel-bc-back-to-top"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/click-copy/" target="_blank" class="heading-element-setting">Click & Copy</a>
										<input type="checkbox" id="panel-bc-clickandcopy" name="panelbcclickandcopy" value="true" <?php checked('true', $bc_clickandcopy_enable); ?>/>
										<label for="panel-bc-clickandcopy"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/codepen/" target="_blank" class="heading-element-setting">Codepen</a>
										<input type="checkbox" id="panel-bc-codepen" name="panelbccodepen" value="true" <?php checked('true', $bc_codepen_enable); ?>/>
										<label for="panel-bc-codepen"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/dark-mode/" target="_blank" class="heading-element-setting">Dark Mode</a>
										<input type="checkbox" id="panel-bc-dark-mode" name="panelbcdarkmode" value="true" <?php checked('true', $bc_dark_mode_enable); ?>/>
										<label for="panel-bc-dark-mode"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/expanders/" target="_blank" class="heading-element-setting">Expander</a>
										<input type="checkbox" id="panel-bc-expander" name="panelbcexpander" value="true" <?php checked('true', $bc_expander_enable); ?>/>
										<label for="panel-bc-expander"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/flipbox/" target="_blank" class="heading-element-setting">Flipbox</a>
										<input type="checkbox" id="panel-bc-flipbox" name="panelbcflipbox" value="true" <?php checked('true', $bc_flipbox_enable); ?>/>
										<label for="panel-bc-flipbox"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/glitchy/" target="_blank" class="heading-element-setting">Glitchy</a>
										<input type="checkbox" id="panel-bc-glitchy" name="panelbcglitchy" value="true" <?php checked('true', $bc_glitchy_enable); ?>/>
										<label for="panel-bc-glitchy"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/glowing-card/" target="_blank" class="heading-element-setting">Glowing Card</a>
										<input type="checkbox" id="panel-bc-glowingcard" name="panelbcglowingcard" value="true" <?php checked('true', $bc_glowingcard_enable); ?>/>
										<label for="panel-bc-glowingcard"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/horizontal-marquee/" target="_blank" class="heading-element-setting">Horizontal Marquee</a>
										<input type="checkbox" id="panel-bc-marquee" name="panelbcmarquee" value="true" <?php checked('true', $bc_marquee_enable); ?>/>
										<label for="panel-bc-marquee"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/horizontal-scroll" target="_blank" class="heading-element-setting">Horizontal scroll</a>
										<input type="checkbox" id="panel-next-horizontal-scroll" name="panelnexthorizontalscroll" value="true" <?php checked('true', $next_horizontal_scroll_enable); ?>/>
										<label for="panel-next-horizontal-scroll"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/lamp/" target="_blank" class="heading-element-setting">Lamp</a>
										<input type="checkbox" id="panel-next-lamp" name="panelnextlamp" value="true" <?php checked('true', $next_lamp_enable); ?>/>
										<label for="panel-next-lamp"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/lottie/" target="_blank" class="heading-element-setting">Lottie</a>
										<input type="checkbox" id="panel-bc-lottie" name="panelbclottie" value="true" <?php checked('true', $bc_lottie_enable); ?>/>
										<label for="panel-bc-lottie"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/motion-divider/" target="_blank" class="heading-element-setting">Motion Divider</a>
										<input type="checkbox" id="panel-bc-divider" name="panelbcdivider" value="true" <?php checked('true', $bc_divider_enable); ?>/>
										<label for="panel-bc-divider"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/next-tabs/" target="_blank" class="heading-element-setting">Next Tabs</a>
										<input type="checkbox" id="panel-bc-coretabs" name="panelbccoretabs" value="true" <?php checked('true', $bc_core_tabs_enable); ?>/>
										<label for="panel-bc-coretabs"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/observer/" target="_blank" class="heading-element-setting">Observer</a>
										<input type="checkbox" id="panel-bc-observer" name="panelbcobserver" value="true" <?php checked('true', $bc_observer_enable); ?>/>
										<label for="panel-bc-observer"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/parallax-hover/" target="_blank" class="heading-element-setting">Parallax Hover</a>
										<input type="checkbox" id="panel-bc-3dhover" name="panelbc3dhover" value="true" <?php checked('true', $bc_3dhover_enable); ?>/>
										<label for="panel-bc-3dhover"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/scrollbar/" target="_blank" class="heading-element-setting">Scrollbar</a>
										<input type="checkbox" id="panel-bc-scrollbar" name="panelbcscrollbar" value="true" <?php checked('true', $bc_scrollbar_enable); ?>/>
										<label for="panel-bc-scrollbar"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/sharer/" target="_blank" class="heading-element-setting">Sharer</a>
										<input type="checkbox" id="panel-bc-sharer" name="panelbcsharer" value="true" <?php checked('true', $bc_sharer_enable); ?>/>
										<label for="panel-bc-sharer"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/smooth-scroll/" target="_blank" class="heading-element-setting">Smooth Scroll</a>
										<input type="checkbox" id="panel-bc-smooth-scroll" name="panelbcsmoothscroll" value="true" <?php checked('true', $bc_smooth_scroll_enable); ?>/>
										<label for="panel-bc-smooth-scroll"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/spinner/" target="_blank" class="heading-element-setting">Spinner</a>
										<input type="checkbox" id="panel-bc-spinner" name="panelbcspinner" value="true" <?php checked('true', $bc_spinner_enable); ?>/>
										<label for="panel-bc-spinner"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/table-of-contents" target="_blank" class="heading-element-setting">Table of contents</a>
										<input type="checkbox" id="panel-next-table-of-contents" name="panelnexttableofcontents" value="true" <?php checked('true', $next_table_of_contents_enable); ?>/>
										<label for="panel-next-table-of-contents"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/timeline" target="_blank" class="heading-element-setting">Timeline</a>
										<input type="checkbox" id="panel-next-timeline" name="panelnexttimeline" value="true" <?php checked('true', $next_timeline_enable); ?>/>
										<label for="panel-next-timeline"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/tippy/" target="_blank" class="heading-element-setting">Tippy</a>
										<input type="checkbox" id="panel-bc-tippy" name="panelbctippy" value="true" <?php checked('true', $bc_tippy_enable); ?>/>
										<label for="panel-bc-tippy"></label>
									</div>

									<div class="element-setting-wrapper">
										<a href="https://nextbricks.io/vertical-marquee/" target="_blank" class="heading-element-setting">Vertical Marquee</a>
										<input type="checkbox" id="panel-bc-infinite-title" name="panelbcinfinitetitle" value="true" <?php checked('true', $bc_infinite_title_enable); ?>/>
										<label for="panel-bc-infinite-title"></label>
									</div>
								</div>
							</div>
						</div>
					</div>	

					<div data-tab="next-ui" class="bc-settings
					<?php echo $active_tab === 'next-ui' ? 'bc-settings--active' : ''; ?>">
						<div class="bc-settings__wrapper">
							<div class="bc-settings__title-save-wrapper">
								<div class="bc-settings__title-wrapper">
									<img class="bc-settings__icon" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/nextbricks_nextui.png'?>">
									<span class="bc-settings__title has-span">
										Next UI
									</span>
								</div>

								<div class="bc-settings__save">
									<button type="submit" class="panel-bc-submit">Save</button>
								</div>
							</div>

							<div class="wrapper-element-setting-wrapper">
								<input type="hidden" name="bc-dashboard-form"/>

								<div class="search-toggle-all-wrapper">
									<div class="element-setting-wrapper bc-feature-core-ui">
										<span class="heading-element-setting">Enable Next UI</span>
										<input type="checkbox" id="panel-bc-core-ui" name="panelbccoreui" value="true" <?php checked('true', $bc_core_ui_enable); ?>/>
										<label for="panel-bc-core-ui"></label>
									</div>
								</div>

								<div class="element-settings-grid">
									<div class="element-setting-wrapper">
										<span class="heading-element-setting">Edit elements at right panel</span>
										<input type="checkbox" id="panel-bc-left-structure" name="panelbcleftstructure" value="true" <?php checked('true', $bc_core_ui_left_structure_enable); ?>/>
										<label for="panel-bc-left-structure"></label>
									</div>

									<div class="element-setting-wrapper core-ui-custom-logo">
										<span class="heading-element-setting">Logo (17x17)</span>
										<img id="bc-selected-logo" src="<?php echo get_option('bc_core_ui_logo') ?>">
										<button class="bc-logo-button heading-element-remove">Remove</button>
										<button class="bc-logo-button" id="panel-bc-image-upload" name="panelbcimageupload">Upload</button>
										<input type="hidden" id="panel-bc-core-ui-logo" name="panelbccoreuilogo" value="<?php echo get_option('bc_core_ui_logo') ?>"/>
									</div>
									<script>
										var wp_media_dialog_field;
										function selectMedia() {
											var custom_uploader;
											if (custom_uploader) {
												custom_uploader.open();
												return;
											}
											custom_uploader = wp.media.frames.file_frame = wp.media({
												title: 'Choose Image',
												button: {
													text: 'Choose Image'
												},
												multiple: false
											});
											custom_uploader.on('select', function() {
												attachment = custom_uploader.state().get('selection').first().toJSON();
												wp_media_dialog_field.val(attachment.url);
												wp_media_img_field.attr('src', attachment.url).trigger('change');
											});
											custom_uploader.open();
										}

										jQuery('#panel-bc-image-upload').click(function(e) {
											e.preventDefault();
											wp_media_dialog_field = jQuery('#panel-bc-core-ui-logo');
											wp_media_img_field = jQuery('#bc-selected-logo');
											selectMedia();
										});

										checkAndShowHeading();
										jQuery('#bc-selected-logo').on('change', function() {
											checkAndShowHeading();
										});
										function checkAndShowHeading() {
											var srcAttribute = jQuery('#bc-selected-logo').attr('src');

											if (srcAttribute) {
												jQuery('#bc-selected-logo').show();
												jQuery('.heading-element-remove').show();
											} else {
												jQuery('#bc-selected-logo').hide();
												jQuery('.heading-element-remove').hide();
											}
										}

										jQuery('.heading-element-remove').click(function(e) {
											e.preventDefault();
											jQuery('#panel-bc-core-ui-logo').val("");
											jQuery('#bc-selected-logo').attr('src', "").trigger('change');
										});
									</script>

									<div class="element-setting-wrapper core-ui-custom-color">
										<span class="heading-element-setting">Accent color</span>
										<input type="color" id="panel-bc-core-ui-color" name="panelbccoreuicolor" value="<?php echo esc_attr(get_option('bc_core_ui_color', '#0096FE')) ?>">
										<span class="delete-color-btn bc-logo-button">Remove</span>
									</div>
									<script>
										jQuery('.delete-color-btn').click(function(e) {
											jQuery('#panel-bc-core-ui-color').val("#0096FE");
										});
									</script>
								<div class="element-setting-wrapper">
									<span class="heading-element-setting">WordPress link</span>
									<input id="panel-bc-core-ui-link" name="panelbccoreuiurl" class="bc-text-input bc-text-input--large" value="<?php echo esc_attr(get_option('bc_core_ui_link', '')) ?>" placeholder="Custom URL">
								</div>
								</div>

								<p class="bc-settings__paragraph">
									<i>Latest Bricks version supported: v<?php echo LATEST_BRICKS_VERSION?></i>
									<br>
								</p>
							</div>
						</div>
					</div>

					<div data-tab="animator" class="bc-settings
					<?php echo $active_tab === 'animator' ? 'bc-settings--active' : ''; ?>">
						<div class="bc-settings__wrapper">
							<div class="bc-settings__title-save-wrapper">
								<div class="bc-settings__title-wrapper">
									<img class="bc-settings__icon" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/nextbricks_animator.png'?>">
									<span class="bc-settings__title has-span">
										Animator
									</span>
								</div>

								<div class="bc-settings__save">
									<button type="submit" class="panel-bc-submit">Save</button>
								</div>
							</div>

							<div class="wrapper-element-setting-wrapper">
								<input type="hidden" name="bc-dashboard-form"/>

								<div class="search-toggle-all-wrapper">
									<div class="element-setting-wrapper bc-feature-core-animator">
										<span class="heading-element-setting">Enable Animator</span>
										<input type="checkbox" id="panel-bc-core-animator" name="panelbccoreanimator" value="true" <?php checked('true', $bc_core_animator_enable); ?>/>
										<label for="panel-bc-core-animator"></label>
									</div>
								</div>

								<div class="element-settings-grid">
									<div class="element-setting-wrapper core-ui-custom-color">
										<span class="heading-element-setting">Accent color</span>
										<input type="color" id="panel-bc-core-animator-color" name="panelbccoreanimatorcolor" value="<?php echo esc_attr(get_option('bc_core_animator_color', '#0096FE')) ?>">
										<span class="delete-animator-color-btn bc-logo-button">Remove</span>
									</div>

									<div class="element-setting-wrapper">
										<span class="heading-element-setting">Shortcut</span>
										<div class="heading-element-setting">
											Ctrl + Shift +
											<input id="panel-bc-core-animator-shortcut" class="bc-text-input" name="panelbccoreanimatorshortcut" value="<?php echo esc_attr(get_option('bc_core_animator_shortcut', 'A')) ?>">
										</div>
									</div>
								</div>
								<script>
									jQuery('.delete-animator-color-btn').click(function(e) {
										jQuery('#panel-bc-core-animator-color').val("#0096FE");
									});
								</script>
							</div>
						</div>
					</div>

					<div data-tab="pagetransitions" class="bc-settings
					<?php echo $active_tab === 'pagetransitions' ? 'bc-settings--active' : ''; ?>">
						<div class="bc-settings__wrapper">
							<div class="bc-settings__title-save-wrapper">
								<div class="bc-settings__title-wrapper">
									<img class="bc-settings__icon" src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/nextbricks_pagetransitions.png'?>">
									<span class="bc-settings__title has-span">
										Page Transitions
									</span>
								</div>

								<div class="bc-settings__save">
									<button type="submit" class="panel-bc-submit">Save</button>
								</div>
							</div>

							<div class="wrapper-element-setting-wrapper">
								<input type="hidden" name="bc-dashboard-form"/>

								<div class="search-toggle-all-wrapper">
									<div class="element-setting-wrapper bc-feature-core-animator">
										<span class="heading-element-setting">Enable Page Transitions</span>
										<input type="checkbox" id="panel-bc-pagetransitions" name="panelbcpagetransitions" value="true" <?php checked('true', $bc_pagetransitions_enable); ?>/>
										<label for="panel-bc-pagetransitions"></label>
									</div>
								</div>

								<div class="element-settings-grid">
									<div class="element-setting-wrapper core-ui-custom-color">
										<span class="heading-element-setting">Accent color</span>
										<input type="color" id="panel-bc-pagetransitions-color" name="panelbcpagetransitionscolor" value="<?php echo esc_attr(get_option('bc_pagetransitions_color', '#0096FE')) ?>">
										<span class="delete-pagetransitions-color-btn bc-logo-button">Remove</span>
									</div>

									<div class="element-setting-wrapper">
										<span class="heading-element-setting">Shortcut</span>
										<div class="heading-element-setting">
											Ctrl + Shift +
											<input id="panel-bc-pagetransitions-shortcut" class="bc-text-input" name="panelbcpagetransitionsshortcut" value="<?php echo esc_attr(get_option('bc_pagetransitions_shortcut', 'P')) ?>">
										</div>
									</div>
								</div>
								<script>
									jQuery('.delete-pagetransitions-color-btn').click(function(e) {
										jQuery('#panel-bc-pagetransitions-color').val("#0096FE");
									});
								</script>

								<div class="element-settings-grid">
									<div class="element-setting-wrapper pt-textarea">
										<span class="heading-element-setting">JavaScript to trigger after transitions:</span>
										<div class="heading-element-setting pt-textarea-wrapper">
										<textarea id="panel-bc-pagetransitions-customjs" name="panelbcpagetransitionscustomjs" class="ui-autocomplete-input" autocomplete="off" role="textbox" aria-autocomplete="list" aria-haspopup="true"><?php echo stripslashes(get_option('bc_pagetransitions_customjs', '')); ?></textarea>
										</div>
									</div>
								</div>
							</div>
						</div>
					</div>
					
					<div data-tab="assets" class="bc-settings
					<?php echo $active_tab === 'assets' ? 'bc-settings--active' : ''; ?>">
						<div class="bc-settings__wrapper">
							<div class="bc-settings__title-save-wrapper">
								<div class="bc-settings__title-wrapper">
									<img src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/dashboard_sidebar_assets.png'?>">
									<span class="bc-settings__title">Assets</span>
								</div>

								<div class="bc-settings__save">
									<button type="submit" class="panel-bc-submit">Save</button>
								</div>
							</div>

							<p class="bc-settings__paragraph">
								Handling libraries can be specially useful to improve performance. 
								You should disable a library if the elements you are using don't require it or if it is already loaded by your own way or other 3rd party addons in order to manage the duplicity.<br>
								When disabling a library be sure it is already loaded and it isn't a deprecated version otherwise elements requiring it won't work as expected.<br>
							</p>
							<div class="wrapper-element-setting-wrapper">
								<input type="hidden" name="bc-dashboard-form"/>

								<div class="element-settings-grid">
									<div class="element-setting-wrapper">
										<span class="heading-element-setting">GSAP</span>
										<input type="checkbox" id="panel-bc-gsap" name="panelbcgsap" value="true" <?php checked('true', $bc_gsap_enable); ?>/>
										<label for="panel-bc-gsap"></label>
									</div>

									<div class="element-setting-wrapper">
										<span class="heading-element-setting">ScrollTrigger</span>
										<input type="checkbox" id="panel-bc-scrolltrigger" name="panelbcscrolltrigger" value="true" <?php checked('true', $bc_scrolltrigger_enable); ?>/>
										<label for="panel-bc-scrolltrigger"></label>
									</div>

									<div class="element-setting-wrapper">
										<span class="heading-element-setting">SplitText</span>
										<input type="checkbox" id="panel-bc-splittext" name="panelbcsplittext" value="true" <?php checked('true', $bc_splittext_enable); ?>/>
										<label for="panel-bc-splittext"></label>
									</div>
								</div>
							</div>
						</div>
					</div>
				</form>

				<div data-tab="license" class="bc-settings <?php echo $active_tab === 'license' ? 'bc-settings--active' : ''; ?>">
					<div class="bc-settings__wrapper">
						<div class="bc-settings__title-save-wrapper">
							<div class="bc-settings__title-wrapper">
								<img src="<?php echo NEXTBRICKS_PLUGIN_URL . 'licensing/srcs/dashboard_sidebar_license.png'?>">
								<span class="bc-settings__title" data-license-activated="<?php echo get_option('activate_brickscore')?>">
									License
									<span class="bc-sidebar__license-span-inactive">Inactive</span>
									<span class="bc-sidebar__license-span-active">Active</span>
								</span>
							</div>
						</div>

						<div class="bc-license">
							<?php settings_errors(); ?>

							<div class="<?php echo esc_attr($this->name) . '-form-container'; ?> wrapper-form-container">
								<form method="post" action="<?php echo esc_attr($this->form_action_url()); ?>">
									<input type="hidden" name="_action" value="<?php echo esc_attr($action); ?>">
									<input type="hidden" name="_nonce" value="<?php echo esc_attr(wp_create_nonce($this->client->name)); ?>">
									<input type="hidden" name="activation_id" value="<?php echo esc_attr($this->activation_id); ?>">

									<div class="bc-form__wrapper">
										<?php if ('activate' === $action) : ?> 
											<input placeholder="License" class="bc-form__input" type="password" autocomplete="off" name="license_key" id="license_key" value="<?php echo esc_attr($this->license_key); ?>" autofocus>
											<?php
											update_option('activate_brickscore', false);
											?>
										<?php else : ?>
											<div class="label-license-deactivate-state">This site is activated</div>
											<?php update_option('activate_brickscore', true); ?>
										<?php endif; ?>

										<?php if (isset($_GET['debug'])) : // phpcs:ignore  ?>
											<label for="license_id"><?php echo esc_html(sprintf($this->client->__('License ID', 'surecart'), $this->client->name)); ?></label>
											<input class="widefat" type="text" autocomplete="off" name="license_id" id="license_id" value="<?php echo esc_attr($this->license_id); ?>" autofocus>

											<label for="activation_id"><?php echo esc_html(sprintf($this->client->__('Activation ID', 'surecart'), $this->client->name)); ?></label>
											<input class="widefat" type="text" autocomplete="off" name="activation_id" id="activation_id" value="<?php echo esc_attr($this->activation_id); ?>" autofocus>
										<?php endif; ?>

										<?php submit_button('activate' === $action ? $this->client->__('Activate') : $this->client->__('Deactivate')); ?>
									</div>
								</form>
							</div>
						</div>
					</div>
				</div>
			</div> <!--Close bc-sidebar-settings-wrapper-->
		</div> <!--Close bc-body-->

		<?php
	}

	function get_data_from_active_tab_script() {
		if (isset($_POST['bc_active_tab'])) {
			update_option('bc_active_tab', sanitize_text_field($_POST['bc_active_tab']));
		}
		wp_die();
	}

	/**
	 * Print the css for the form.
	 *
	 * @return void
	 */
	public function print_css() {
		?>
		<script>
			/*switches*/
			document.addEventListener('DOMContentLoaded', function () {
				var tabs = document.querySelectorAll('.bc-settings');
				
				tabs.forEach(function(tab) {
					var toggleAll = tab.querySelector('.toggle-all');
					var individualSwitches = tab.querySelectorAll('.wrapper-element-setting-wrapper input[type="checkbox"]');

					if(toggleAll){
						toggleAll.addEventListener('change', function () {
							var isChecked = toggleAll.checked;
							individualSwitches.forEach(function (switchElement) {
								switchElement.checked = isChecked;
							});
						});
					}
				});
			});
		</script>
		<script>
			/*search*/
			document.addEventListener('DOMContentLoaded', function () {
				var tabs = document.querySelectorAll('.bc-settings');
				
				tabs.forEach(function(tab) {
					var searchInput = tab.querySelector('#searchInput');

					if(searchInput){
						searchInput.addEventListener('keyup', function() {
							var searchQuery = this.value.toLowerCase();
							var elements = tab.querySelectorAll('.element-setting-wrapper:not(.toggle-all-wrapper)');

							elements.forEach(function(element) {
								if (element.textContent.toLowerCase().indexOf(searchQuery) === -1) {
									element.classList.add('element-setting-wrapper--hidden');
								} else {
									element.classList.remove('element-setting-wrapper--hidden');
								}
							});
						});
					}
				});
			});
		</script>
		<?php
			$bc_active_tab = get_option('bc_active_tab', 'texts');

			wp_register_script( 'bc_active_tab_script', plugins_url('change_active_tab.js', __FILE__) );
			wp_localize_script('bc_active_tab_script', 'data_tab', array(
				'tab' => $bc_active_tab,
				'ajaxurl' => admin_url('admin-ajax.php'),
			));
			wp_enqueue_script('bc_active_tab_script');
		?>
		<style>
		/*Page Transitions textarea*/
		.element-setting-wrapper.pt-textarea{
			grid-column: span 3;
			flex-direction: column;
			align-items: flex-start;
			row-gap: 12px;
		}
		.pt-textarea-wrapper{
			width: 100%;
		}
		.pt-textarea-wrapper textarea{
			width: 100%;
			color: black;
			border-radius: 8px;
			background: lightgrey;
			padding: 8px;

			outline: none !important;
			border-color: unset !important;
			box-shadow: none !important;
		}
		</style>
		<style>
		a.heading-element-setting:hover{
			text-decoration: underline;
		}
		/*text input*/
		.bc-text-input{
			all: unset;
			width: 30px;

			color: black;
			border-radius: 4px;
			background: lightgrey;

			text-align: center;
		}
		.bc-text-input--large{
			width: 120px;
			height: 24px;
			padding: 0 5px !important;
			text-align: left;
		}
		</style>
		<style>
		/*Tabs*/
		.bc-settings:not(.bc-settings--active){
			display: none;
		}
		.no-license-form:not(:has(.bc-settings--active)){
			display: none;
		}
		</style>
		<style>
		/*wordpress elements*/
		#wpfooter{
			display: none;
		}
		#wpcontent{
			background: #111111;
			padding-right: 20px;
		}
		#wpbody-content{
			padding-bottom: 0;
		}

		/*Dashboard styles*/
		.bc-body,
		.bc-body *{
			box-sizing: border-box;
			padding: 0;
			margin: 0;

			font-family: "Inter";
			letter-spacing: -0.022em;
		}
		.bc-body *:focus {
			box-shadow: unset !important;
		}
		.bc-body *:focus-visible {
			outline: thin dashed #5B5B5B;
		}
		.bc-body a{
			color: unset;
			text-decoration: unset;
		}
		.bc-body{
			width: 100%;

			display: flex;
    		justify-content: center;
			align-items: center;

			min-height: 675px;
			height: calc(100vh - 32px);
			padding: 20px 0;
		}
		body:has(#uip-admin-menu) .bc-body{
			height: 100vh !important;
		}
		/*sidebar*/
		.bc-sidebar-settings-wrapper{
			background: #111111;
			border: 1px solid #26262A;
			box-shadow: 0px 5px 0px 0px rgba(0,0,0,0.5);

			width: 100%;
			max-width: 1080px;
			min-height: 635px;
			height: 100%;
			max-height: 992px;
			border-radius: 12px;

			display: grid;
			grid-template-columns: 0.5fr 1.5fr;
			overflow: hidden;
		}
		.bc-sidebar{
			border-right: 1px solid #26262A;

			display: flex;
			flex-direction: column;
			justify-content: space-between;
			row-gap: 5px;

			padding: 16px;
			padding-top: 8px;
		}
		.bc-sidebar__caption{
			font-size: 12px;
			color: #CCCCCC;

			margin: 12px 8px;
		}
		.bc-sidebar__top-wrapper{
			display: flex;
    		flex-direction: column;
			row-gap: 4px;
		}
		.bc-sidebar__bottom-wrapper{
			display: flex;
    		flex-direction: column;
			row-gap: 4px;
		}
		.bc-sidebar__nav-item-separator{
			height: 1px;
			width: 100%;
			background: #26262A;
		}
		.bc-sidebar__nav-item-separator--mobile{
			display: none;
		}
		.bc-sidebar__nav-item-wrapper{
			display: flex;
			align-items: center;
			column-gap: 12px;

			padding: 6px;
			border-radius: 12px;
			height: 44px;

			color: #CCCCCC !important;
			font-weight: 400;
			line-height: 0;

			cursor: pointer;
			transition: background .2s, color .2s;
		}
		.has-span{
			position: relative;
		}
		.has-span span{
			position: absolute;
			top: -5px;
			left: calc(100% + 3px);

			color: #F03E6B;
			font-size: 10px;
			letter-spacing: 0;
		}
		.bc-sidebar__nav-item-wrapper:hover,
		.bc-sidebar__nav-item-wrapper-active{
			background: #2B2B2B;
			box-shadow: inset 0px 0px 2px 0px rgba(163, 163, 163, 0.25);
			color: #fff !important;
		}
		.bc-sidebar__nav-top-item-wrapper{
			column-gap: 8px;
			padding: 12px 4px;
			white-space: nowrap;
		}
		.bc-sidebar__nav-item-image{
			margin-left: 5px;
			width: 20px;
			height: 20px;
		}
		.bc-sidebar__nav-item-image--icon{
			width: 17px;
			height: 17px;
		}
		.bc-sidebar__logo-wrapper{
			display: flex;
			align-items: center;
			column-gap: 12px;

			padding: 10px 4px;
			padding-bottom: 4px;

			color: #CCCCCC !important;
			font-weight: 400;
		}
		.bc-sidebar__logo{
			margin-left: 5px;
			width: 20px;
			height: 20px;
		}
		.bc-sidebar__version:hover{
			text-decoration: underline;
		}
		.bc-sidebar__license-span{
			position: relative
		}
		.bc-sidebar__license-span-inactive,
		.bc-sidebar__license-span-active{
			display: none;

			position: absolute;
			top: -5px;
			left: calc(100% + 2px);
			font-size: 10px;
			letter-spacing: 0;
		}
		.bc-body:has(.label-license-deactivate-state) .bc-sidebar__license-span-active,
		.bc-sidebar__license-span[data-license-activated="1"] .bc-sidebar__license-span-active{
			display: block;
			color: #47FF91;
		}
		.bc-sidebar__license-span[data-license-activated] .bc-sidebar__license-span-inactive{
			display: block;
			color: #F03E6B;
		}
		.bc-body:has(.label-license-deactivate-state) .bc-sidebar__license-span-inactive,
		.bc-sidebar__license-span[data-license-activated="1"] .bc-sidebar__license-span-inactive,
		.bc-body:has(.bc-form__input) .bc-sidebar__license-span-active{
			display: none;
		}
		.bc-body:has(.bc-form__input) .bc-sidebar__license-span-inactive{
			display: block;
			color: #F03E6B;
		}
		/*Settings*/
		.bc-sidebar,
		.no-license-form{
			overflow: auto;
		}
		.bc-settings{
			padding: 32px;

			display: flex;
			flex-direction: column;
			row-gap: 32px;
		}
		.bc-settings__wrapper{
			display: flex;
			flex-direction: column;
			row-gap: 32px;
		}
		.bc-sidebar::-webkit-scrollbar,
		.no-license-form::-webkit-scrollbar{
			width: 4px;
		}
		.bc-sidebar::-webkit-scrollbar-thumb,
		.no-license-form::-webkit-scrollbar-thumb{
			border-radius: 10px;
			background-color: #454545;
		}
		.bc-settings__title{
			position: relative;
		}
		.bc-settings__title-save-wrapper{
			display: flex;
			justify-content: space-between;
			align-items: center;
		}
		.bc-settings__title-wrapper{
			display: flex;
			align-items: center;
			column-gap: 8px;

			color: #CCCCCC !important;
			font-weight: 400;
			line-height: 0;

			height: 20px;
		}
		.bc-settings__icon{
			margin-left: 5px;
			width: 20px;
			height: 20px;
		}
		.bc-settings__save button{
			all: unset;
			cursor: pointer;

			background: #0096FE;
			color: #fff;

			padding: 4px 10px;
			border-radius: 8px;
		}
		.wrapper-element-setting-wrapper{
			display: flex;
			flex-direction: column;
			row-gap: 16px;
		}
		.element-setting-wrapper{
			background: #2b2b2b;
    		box-shadow: inset 0px 0px 2px 0px rgba(163, 163, 163, 0.25);

			border-radius: 12px;
			padding: 12px;

			display: flex;
			justify-content: space-between;
			align-items: center;
			column-gap: 8px;
		}
		.element-setting-wrapper--hidden{
			display: none;
		}
		.search-toggle-all-wrapper{
			display: flex;
			column-gap: 8px;
		}
		.toggle-all-wrapper,
		.bc-feature-core-animator,
		.bc-feature-core-ui{
			max-width: fit-content;
			padding: 8px;
			column-gap: 12px;
		}
		#searchInput{
			all: unset;
			box-sizing: border-box;
			padding-block: 0;
			padding-inline: 0;
			background: #2b2b2b !important;
    		box-shadow: inset 0px 0px 2px 0px rgba(163, 163, 163, 0.25) !important;

			border-radius: 12px !important;

			padding: 8px;
			max-width: 120px;

			color: #CCCCCC !important;
		}
		#searchInput::placeholder{
			color: #CCCCCC !important;
		}
		.heading-element-setting{
			color: #CCCCCC !important;
		}
		.element-settings-grid{
			display: grid;
			align-items: start;
			grid-template-columns: 1fr 1fr 1fr;
			gap: 8px;
		}
		.core-ui-custom-logo,
		.core-ui-custom-color{
			padding: 8px;
			padding-left: 12px;
		}
		.bc-logo-button{
			all: unset;
			cursor: pointer;
			background: #0096FE;
			color: #fff;
			padding: 4px 8px;
			border-radius: 8px;
		}
		#panel-bc-core-ui-color,
		#panel-bc-core-animator-color,
		#panel-bc-pagetransitions-color{
			background: unset !important;
    		border: unset !important;
			outline: none !important;
			box-sizing: border-box !important;

			width: 58px !important;
			height: 26px !important;
		}
		#panel-bc-core-ui-color::-webkit-color-swatch,
		#panel-bc-core-animator-color::-webkit-color-swatch,
		#panel-bc-pagetransitions-color::-webkit-color-swatch{
			border-radius: 8px;
			margin-top: -4px;
			margin-right: -2px;
			width: 58px;
			height: 24px;
		}
		/*Settings - Library*/
		.bc-settings__paragraph{
			color: #CCCCCC;
		}
		</style>
		<style>
		/*breakpoints*/
		@media screen and (max-width: 782px) {
			#wpcontent{
				padding-right: 10px;
			}
			.bc-body{
				height: calc(100vh - 46px);
				padding: 10px 0;
			}
			.bc-sidebar-settings-wrapper{
				grid-template-columns: 1fr;
				overflow-y: scroll;
			}
			.bc-sidebar-settings-wrapper::-webkit-scrollbar{
				width: 4px;
			}
			.bc-sidebar-settings-wrapper::-webkit-scrollbar-thumb{
				border-radius: 10px;
				background-color: #454545;
			}
			/*Sidebar*/
			.bc-sidebar {
				border-right: unset;
			}
			.bc-sidebar__nav-item-separator--mobile{
				display: block;
				margin-top: 4px;
			}
			/*Settings*/
			.no-license-form{
				overflow: unset;
			}
			.bc-settings{
				padding: 24px;
			}
			.element-settings-grid{
				grid-template-columns: 1fr;
			}
		}
		</style>
		<style>
			/*License styles*/
			.bc-form__wrapper{
				display: grid;
				gap: 12px;
			}
			#license_key{
				all: unset;
				box-sizing: border-box;
				padding-block: 0;
				padding-inline: 0;
				background: #1E1E1D !important;
				border-radius: 12px !important;
				border: 1px solid #2B2B2B !important;
				padding: 8px !important;
				color: #CCCCCC !important;
			}
			#license_key::placeholder{
				color: #CCCCCC !important;
			}
			p.submit{
				all: unset;
			}
			.submit #submit{
				all: unset;
				cursor: pointer;
				background: #0096FE;
				color: #fff;
				padding: 4px 8px;
				border-radius: 8px;

				font-family: "Inter";
				letter-spacing: -0.022em;
				font-size: 12px;
			}
			.label-license-deactivate-state{
				color: #CCCCCC;
			}
			.bc-license .notice{
				margin-bottom: 8px;
			}
		</style>
		<style>
			/*Core UI tab styles*/
			#bc-selected-logo{
				height: 20px;
				width: 20px;
			}
			.bc-logo-wrapper,
			.color-picker-wrapper{
				display: flex;
				align-items: center;
				justify-content: center;
				column-gap: 15px;
			}
		</style>
		<style>
			/*Library tab styles*/
			.bc-settings__heading{
				font-size: 18px;
				font-weight: 600;
			}
			.bc-settings__icon--menus-library{
				bottom: 0 !important;
			}
			.bc-settings__screenshot{
				width: 100%;
			}
			p{
				font-size: 14px;
			}
		</style>
		<style>
			/*Switch styles*/
			.element-setting-wrapper input[type=checkbox]{
				position: absolute !important;
				height: 0 !important;
				width: 0 !important;
				visibility: hidden !important;
				margin: 0 !important;
				padding: 0 !important;
				min-width: 0 !important;
				min-height: 0 !important;
				max-width: 0 !important;
				max-height: 0 !important;
				border: 0 !important;
			}
			.element-setting-wrapper label {
				cursor: pointer;
				text-indent: -9999px;
				min-width: 32px;
				height: 16px;
				background: #000;
				display: block;
				border-radius: 100px;
				position: relative;
				margin: 0 !important;
				transition: 0.2s ease-in-out;
			}
			.element-setting-wrapper label:after {
				content: '';
				position: absolute;
				top: 0;
				left: 0;
				width: 16px;
				height: 16px;
				background: #fff;
				border-radius: 90px;
				box-shadow: 0 0 12px rgba(0, 0, 0, 0.8);
				transition: 0.2s ease-in-out;
			}
			.element-setting-wrapper input:checked + label {
				background: #18CF61;
			}
			.element-setting-wrapper input:checked + label:after {
				left: 100%;
				transform: translateX(-100%);
			}
		</style>
		<style>
			/*Hide license form in playground*/
			[data-tab="license"]:has(>[src*="instawp.xyz"] ~ [data-license-activated="1"]) {
				display: none !important;
			}
		</style>
		<?php
	}

	/**
	 * Get the activation.
	 *
	 * @return Object|false
	 */
	public function get_activation() {
		$activation = false;
		if ( $this->activation_id ) {
			$activation = $this->client->activation()->get( $this->activation_id );
			if ( is_wp_error( $activation ) ) {
				$this->add_error( 'deactivaed', $this->client->__( 'Your license has been deactivated for this site.', 'surecart' ) );
				$this->clear_options();
			}
		}

		return $activation;
	}

	/**
	 * License form submit
	 */
	public function license_form_submit() {
		// only if we are submitting.
		if ( ! isset( $_POST['submit'] ) ) {
			return;
		}

		// Check nonce.
		if ( ! isset( $_POST['_nonce'], $_POST['_action'] ) ) {
			$this->add_error( 'missing_info', $this->client->__( 'Please add all information' ) );
			return;
		}

		// Cerify nonce.
		if ( ! wp_verify_nonce( $_POST['_nonce'], $this->client->name ) ) {
			$this->add_error( 'unauthorized', $this->client->__( "You don't have permission to manage licenses." ) );
			return;
		}

		// handle activation.
		if ( 'activate' === $_POST['_action'] ) {
			$activated = $this->client->license()->activate( sanitize_text_field( $_POST['license_key'] ) );
			if ( is_wp_error( $activated ) ) {
				$this->add_error( $activated->get_error_code(), $activated->get_error_message() );
				return;
			}

			if ( ! empty( $this->menu_args['activated_redirect'] ) ) {
				$this->redirect( $this->menu_args['activated_redirect'] );
				exit;
			}

			?>
			<script>
				location.reload();
			</script>
			<?php

			return $this->add_success( 'activated', $this->client->__( 'This site was successfully activated.', 'surecart' ) );
		}

		// handle deactivation.
		if ( 'deactivate' === $_POST['_action'] ) {
			$deactivated = $this->client->license()->deactivate( sanitize_text_field( $_POST['activation_id'] ) );
			if ( is_wp_error( $deactivated ) ) {
				$this->add_error( $deactivated->get_error_code(), $deactivated->get_error_message() );
			}

			if ( ! empty( $this->menu_args['deactivated_redirect'] ) ) {
				$this->redirect( $this->menu_args['deactivated_redirect'] );
				exit;
			}

			?>
			<script>
				location.reload();
			</script>
			<?php

			return $this->add_success( 'deactivated', $this->client->__( 'This site was successfully deactivated.', 'surecart' ) );
		}
	}

	/**
	 * Redirect to a url client-side.
	 * We need to do this to avoid "headers already sent" messages.
	 *
	 * @param string $url Url to redirect.
	 *
	 * @return void
	 */
	public function redirect( $url ) {
		?>
		<div class="spinner is-active"></div>
		<script>
			window.location.assign("<?php echo esc_url( $url ); ?>");
		</script>
		<?php
	}

	/**
	 * Add a notice.
	 *
	 * @param string $code Notice code.
	 * @param string $message Notice message.
	 * @param string $type Notice type.
	 *
	 * @return void
	 */
	public function add_notice( $code, $message, $type = 'info' ) {
		add_settings_error(
			$this->name . '_license_options', // matches what we registered in `register_setting.
			$code, // the error code.
			$message,
			$type
		);
	}

	/**
	 * Add an error.
	 *
	 * @param string $code Error code.
	 * @param string $message Error message.
	 *
	 * @return void
	 */
	public function add_error( $code, $message ) {
		$this->add_notice( $code, $message, 'error' );
	}

	/**
	 * Add an success message
	 *
	 * @param string $code Success code.
	 * @param string $message Success message.
	 *
	 * @return void
	 */
	public function add_success( $code, $message ) {
		$this->add_notice( $code, $message, 'success' );
	}

	/**
	 * Set an option.
	 *
	 * @param string $name Name of option.
	 *
	 * @return mixed
	 */
	public function __get( $name ) {
		return $this->get_option( 'sc_' . $name );
	}

	/**
	 * Set an option
	 *
	 * @param string $name Name of option.
	 * @param mixed  $value Value.
	 *
	 * @return bool
	 */
	public function __set( $name, $value ) {
		return $this->set_option( 'sc_' . $name, $value );
	}
}
