<?php 
/**
 * Register/enqueue custom scripts and styles
 */
add_action( 'wp_enqueue_scripts', function() {
	// Enqueue your files on the canvas & frontend, not the builder panel. Otherwise custom CSS might affect builder)
	if ( ! bricks_is_builder_main() ) {
		wp_enqueue_style( 'bricks-child', get_stylesheet_uri(), ['bricks-frontend'], filemtime( get_stylesheet_directory() . '/style.css' ) );
	}
} );

/**
 * Register custom elements
 */
add_action( 'init', function() {
  $element_files = [
    __DIR__ . '/elements/title.php',
  ];

  foreach ( $element_files as $file ) {
    \Bricks\Elements::register_element( $file );
  }
}, 11 );

/**
 * Add text strings to builder
 */
add_filter( 'bricks/builder/i18n', function( $i18n ) {
  // For element category 'custom'
  $i18n['custom'] = esc_html__( 'Custom', 'bricks' );

  return $i18n;
} );









/**
 * bidirectional sync between Projects and Services
 */

add_action('acf/save_post', 'sync_project_service_fields', 20);

function sync_project_service_fields($post_id) {
    // Skip revisions and autosaves
    if (wp_is_post_revision($post_id) || wp_is_post_autosave($post_id)) {
        return;
    }
    
    $post_type = get_post_type($post_id);
    
    // Only run for projects or services
    if (!in_array($post_type, ['projects', 'services'])) {
        return;
    }
    
    // Prevent infinite loops
    remove_action('acf/save_post', 'sync_project_service_fields', 20);
    
    if ($post_type === 'projects') {
        sync_from_project($post_id);
    } elseif ($post_type === 'services') {
        sync_from_service($post_id);
    }
    
    // Re-add the hook
    add_action('acf/save_post', 'sync_project_service_fields', 20);
}

function sync_from_project($project_id) {
    $related_services = get_field('related_service', $project_id) ?: [];
    
    // Get all services to update them
    $all_services = get_posts([
        'post_type' => 'services',
        'posts_per_page' => -1,
        'fields' => 'ids'
    ]);
    
    foreach ($all_services as $service_id) {
        $current_projects = get_field('related_projects', $service_id) ?: [];
        
        // Should this service be linked to this project?
        $should_link = in_array($service_id, wp_list_pluck($related_services, 'ID'));
        
        // Is it currently linked?
        $is_linked = in_array($project_id, wp_list_pluck($current_projects, 'ID'));
        
        if ($should_link && !$is_linked) {
            // Add the project
            $current_projects[] = $project_id;
            update_field('related_projects', $current_projects, $service_id);
        } elseif (!$should_link && $is_linked) {
            // Remove the project
            $current_projects = array_filter($current_projects, function($p) use ($project_id) {
                $pid = is_object($p) ? $p->ID : $p;
                return $pid != $project_id;
            });
            update_field('related_projects', array_values($current_projects), $service_id);
        }
    }
}

function sync_from_service($service_id) {
    $related_projects = get_field('related_projects', $service_id) ?: [];
    
    // Get all projects to update them
    $all_projects = get_posts([
        'post_type' => 'projects',
        'posts_per_page' => -1,
        'fields' => 'ids'
    ]);
    
    foreach ($all_projects as $project_id) {
        $current_services = get_field('related_service', $project_id) ?: [];
        
        // Should this project be linked to this service?
        $should_link = in_array($project_id, wp_list_pluck($related_projects, 'ID'));
        
        // Is it currently linked?
        $is_linked = in_array($service_id, wp_list_pluck($current_services, 'ID'));
        
        if ($should_link && !$is_linked) {
            // Add the service
            $current_services[] = $service_id;
            update_field('related_service', $current_services, $project_id);
        } elseif (!$should_link && $is_linked) {
            // Remove the service
            $current_services = array_filter($current_services, function($s) use ($service_id) {
                $sid = is_object($s) ? $s->ID : $s;
                return $sid != $service_id;
            });
            update_field('related_service', array_values($current_services), $project_id);
        }
    }
}


/**
 * Get the current Bricks Builder loop index.
 * Returns formatted numbers (01, 02, 03…).
 */
// function bricks_loop_index_number() {
//     $index = Bricks\Query::get_loop_index();

//     if ($index !== null) {
//         // Add +1 because loop index starts from 0
//         // str_pad adds a leading zero (01, 02, 03...)
//         return str_pad($index + 1, 2, '0', STR_PAD_LEFT);
//     }

//     return null;
// }

// /**
//  * Allow custom functions to be used with {echo:...} in Bricks.
//  */
// add_filter( 'bricks/code/echo_function_names', function() {
//   return [
//     'bricks_loop_index_number',
//   ];
// } ); 





/**
 * Minimal hierarchical numbering system
 */
global $bricks_hierarchy_counter;
if (!isset($bricks_hierarchy_counter)) {
    $bricks_hierarchy_counter = [
        'parent_counts' => [],
        'parent_counter' => 0  // Add dedicated parent counter
    ];
}

/**
 * Get hierarchical numbering for services or posts
 */
function bricks_hierarchical_number() {
    global $bricks_hierarchy_counter;
    
    $current_post = get_post();
    if (!$current_post) {
        return null;
    }
    
    $post_parent = $current_post->post_parent;
    $current_index = Bricks\Query::get_loop_index();
    
    if ($current_index === null) {
        return null;
    }
    
    // Reset counter at the beginning of each loop
    if ($current_index === 0) {
        $bricks_hierarchy_counter['parent_counter'] = 0;
        $bricks_hierarchy_counter['parent_counts'] = [];
    }
    
    // If this is a top-level item (no parent)
    if (!$post_parent) {
        // Increment the dedicated parent counter
        $bricks_hierarchy_counter['parent_counter']++;
        $parent_number = $bricks_hierarchy_counter['parent_counter'];
        
        // Store this parent's number for its children to reference
        $bricks_hierarchy_counter['parent_counts'][$current_post->ID] = $parent_number;
        return (string) $parent_number;
    }
    
    // This is a child item
    $parent_number = $bricks_hierarchy_counter['parent_counts'][$post_parent] ?? 1;
    $child_position = bricks_get_child_position($current_post->ID, $post_parent);
    
    return $parent_number . '.' . $child_position;
}

/**
 * Get parent item title
 */
function bricks_get_parent_title() {
    $current_post = get_post();
    if (!$current_post || !$current_post->post_parent) {
        return '';
    }
    
    $parent = get_post($current_post->post_parent);
    return $parent ? $parent->post_title : '';
}

/**
 * Helper function to get child position among siblings
 */
function bricks_get_child_position($post_id, $parent_id) {
    static $child_positions = [];
    
    $cache_key = $parent_id . '_children';
    
    if (!isset($child_positions[$cache_key])) {
        $children = get_posts([
            'post_parent' => $parent_id,
            'post_type' => get_post_type($post_id),
            'orderby' => 'date',
            'order' => 'ASC',
            'posts_per_page' => -1,
            'post_status' => 'publish'
        ]);
        
        $position = 1;
        foreach ($children as $child) {
            $child_positions[$cache_key][$child->ID] = $position++;
        }
    }
    
    return $child_positions[$cache_key][$post_id] ?? 1;
}

/**
 * Register functions for Bricks echo
 */
add_filter('bricks/code/echo_function_names', function() {
    return [
        'bricks_hierarchical_number',
        'bricks_get_parent_title',
    ];
});









// Force permalink to /service/{post_name}/
function custom_services_permalink($post_link, $post) {
    if ($post->post_type === 'services') {
        return home_url('service/' . $post->post_name . '/');
    }
    return $post_link;
}
add_filter('post_type_link', 'custom_services_permalink', 10, 2);

// Rewrite rule so WP matches the new structure
function custom_services_rewrite_rules() {
    add_rewrite_rule('^service/([^/]+)/?$', 'index.php?post_type=services&name=$matches[1]', 'top');
}
add_action('init', 'custom_services_rewrite_rules');

